/**
 * Copyright © 2014-2015 XebiaLabs B.V. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.booter.remote;

import java.util.*;

import org.jboss.resteasy.client.ClientExecutor;
import org.jboss.resteasy.client.ProxyFactory;
import org.jboss.resteasy.spi.ResteasyProviderFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.booter.remote.service.StreamingImportingService;
import com.xebialabs.deployit.booter.remote.resteasy.DeployitEntityExtractorFactory;
import com.xebialabs.deployit.core.api.ConfigurationService;
import com.xebialabs.deployit.engine.api.*;

import static org.jboss.resteasy.client.ProxyFactory.createUri;

/**
 * The {@link com.xebialabs.deployit.booter.remote.Proxies} object is used to access services using RESTEasy.
 * The exposed services must never be cached, because on a re-login in {@link HttpClientHolder} the proxies will be invalidated.
 */
public class Proxies {
    private final Map<Class<?>, Object> registeredProxies = new HashMap<Class<?>, Object>();
    private HttpClientHolder httpClientHolder;
    private BooterConfig booterConfig;

    /**
     * Use the second constructor
     */
    @Deprecated
    public Proxies(DeployitCommunicator communicator) {
        this.httpClientHolder = communicator.getHttpClientHolder();
        this.booterConfig = communicator.getConfig();
        init(communicator.getConfig());
    }

    public Proxies(HttpClientHolder httpClientHolder, BooterConfig config) {
        this.httpClientHolder = httpClientHolder;
        this.booterConfig = config;
        init(config);
    }

    private void init(BooterConfig config) {
        for (Class<?> clazz : allProxies()) {
            registerProxy(config.getUrl(), clazz);
        }
    }

    public static Logger getLogger() {
        return logger;
    }

    public void registerProxy(String url, Class<?> clazz) {
        logger.debug("Registering Proxy: {}", clazz);
        ClientExecutor clientExecutor = httpClientHolder.createClientExecutor();
        Map<String, Object> requestAttributes = new HashMap<>();
        requestAttributes.put("BOOTER_CONFIG", booterConfig);
        registeredProxies.put(clazz, ProxyFactory.create(clazz, createUri(url), clientExecutor, ResteasyProviderFactory.getInstance(), new DeployitEntityExtractorFactory(), requestAttributes));
    }

    @SuppressWarnings("unchecked")
    private static Iterable<Class<?>> allProxies() {
        ArrayList<Class<?>> classes = new ArrayList<>();
        Collections.addAll(classes, ServerService.class, MetadataService.class, RepositoryService.class,
                ControlService.class, DeploymentService.class, InspectionService.class, PackageService.class,
                ConfigurationService.class, PermissionService.class, RoleService.class, TaskService.class, TaskBlockService.class, UserService.class);
        return classes;
    }

    @SuppressWarnings("unchecked")
    public <T> T getProxyInstance(Class<T> clazz) {
        return (T) registeredProxies.get(clazz);
    }

    public ServerService getServerService() {
        return getProxyInstance(ServerService.class);
    }

    public MetadataService getMetadataService() {
        return getProxyInstance(MetadataService.class);
    }

    public RepositoryService getRepositoryService() {
        return getProxyInstance(RepositoryService.class);
    }

    public ControlService getControlService() {
        return getProxyInstance(ControlService.class);
    }

    public DeploymentService getDeploymentService() {
        return getProxyInstance(DeploymentService.class);
    }

    public InspectionService getInspectionService() {
        return getProxyInstance(InspectionService.class);
    }

    public PackageService getPackageService() {
        return new StreamingImportingService(httpClientHolder, booterConfig, this);
    }

    public PermissionService getPermissionService() {
        return getProxyInstance(PermissionService.class);
    }

    public RoleService getRoleService() {
        return getProxyInstance(RoleService.class);
    }

    public TaskService getTaskService() {
        return getProxyInstance(TaskService.class);
    }

    public TaskBlockService getTaskBlockService() {
        return getProxyInstance(TaskBlockService.class);
    }

    public UserService getUserService() {
        return getProxyInstance(UserService.class);
    }

    private static final Logger logger = LoggerFactory.getLogger(Proxies.class);

    public static Proxies reinitialize(Proxies oldProxies, HttpClientHolder httpClientHolder, BooterConfig config) {

        Proxies newProxies = new Proxies(httpClientHolder, config);

        if (oldProxies == null) {
            return newProxies;
        }

        for (Class<?> proxyClass : oldProxies.registeredProxies.keySet()) {
            if (!newProxies.registeredProxies.containsKey(proxyClass)) {
                newProxies.registerProxy(config.getUrl(), proxyClass);
            }
        }


        return newProxies;
    }
}
