/**
 * Copyright © 2014-2016 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.booter.remote;

import java.util.*;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistryId;
import com.xebialabs.deployit.plugin.api.reflect.Type;

public class RemoteDescriptorRegistry extends DescriptorRegistry {

    private final ReentrantReadWriteLock readWriteLock = new ReentrantReadWriteLock(true);
    private final Lock readLock = readWriteLock.readLock();
    private final Lock writeLock = readWriteLock.writeLock();

    private Map<Type, Descriptor> descriptors = new HashMap<>();
    private Map<Type, List<Type>> subtypes = new HashMap<>();

    protected RemoteDescriptorRegistry(DescriptorRegistryId id) {
        super(id);
    }

    @Override
    protected boolean isLocal() {
        return false;
    }

    @Override
    protected Collection<Descriptor> _getDescriptors() {
        readLock.lock();
        try {
            return descriptors.values();
        } finally {
            readLock.unlock();
        }
    }

    @Override
    protected Collection<Type> _getSubtypes(Type supertype) {
        readLock.lock();
        try {
            return getFromTypeMap(subtypes, supertype);
        } finally {
            readLock.unlock();
        }
    }

    @Override
    protected Descriptor _getDescriptor(Type type) {
        if (!descriptors.containsKey(type)) {
            throw new IllegalArgumentException(String.format("DescriptorRegistry does not know about type [%s]", type));
        }
        readLock.lock();
        try {
            return descriptors.get(type);
        } finally {
            readLock.unlock();
        }
    }

    @Override
    protected boolean _exists(Type type) {
        readLock.lock();
        try {
            return descriptors.containsKey(type);
        } finally {
            readLock.unlock();
        }
    }

    public static void boot(DeployitCommunicator communicator) {
        RemoteDescriptorRegistry registry = new RemoteDescriptorRegistry(communicator.getConfig());
        DescriptorRegistry.add(registry);
        registry.reboot(communicator);
    }

    public void reboot(DeployitCommunicator communicator) {
        writeLock.lock();
        try {
            List<Descriptor> list = communicator.getProxies().getMetadataService().listDescriptors();
            reboot(list);
        } finally {
            writeLock.unlock();
        }
    }

    public void reboot(List<Descriptor> list) {
        writeLock.lock();
        try {
            Map<Type, Descriptor> newDescriptorsMap = new HashMap<>();
            Map<Type, List<Type>> newSubtypesMap = new HashMap<>();

            for (Descriptor descriptor : list) {
                newDescriptorsMap.put(descriptor.getType(), descriptor);
                for (Type type : descriptor.getSuperClasses()) {
                    addToTypeMap(newSubtypesMap, type, descriptor.getType());
                }
                for (Type type : descriptor.getInterfaces()) {
                    addToTypeMap(newSubtypesMap, type, descriptor.getType());
                }
            }

            this.descriptors = newDescriptorsMap;
            this.subtypes = newSubtypesMap;
        } finally {
            writeLock.unlock();
        }
    }

    public Collection<Descriptor> getLoadedDescriptors() {
        return _getDescriptors();
    }

    public Descriptor getLoadedDescriptor(Type type) {
        return _getDescriptor(type);
    }

    public Descriptor getLoadedDescriptor(String prefixedName) {
        return getLoadedDescriptor(lookupType(prefixedName));
    }

    public Descriptor getLoadedDescriptor(String prefix, String name) {
        return getLoadedDescriptor(lookupType(prefix, name));
    }

    private void addToTypeMap(Map<Type, List<Type>> map, Type key, Type value) {
        List<Type> collection = map.get(key);
        if (null == collection) {
            collection = new ArrayList<>();
        }
        collection.add(value);
        map.put(key, collection);
    }

    private Collection<Type> getFromTypeMap(Map<Type, List<Type>> map, Type key) {
        List<Type> list = map.get(key);
        if (null == list) {
            list = new ArrayList<>();
        }
        return list;
    }
}