/**
 * Copyright © 2014-2016 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.booter.remote.client;

import java.io.File;
import java.io.FileNotFoundException;
import java.net.MalformedURLException;
import java.net.URL;

import com.xebialabs.deployit.engine.api.dto.ImportFromUrlParams;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.booter.remote.DeployitCommunicator;
import com.xebialabs.deployit.booter.remote.service.StreamingImportingService;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;

import static java.lang.String.format;

/**
 * A thin wrapper to close a gap between REST api functionality, provided out of the box by proxies and deployit clients needs.
 */
public class DeployitRemoteClient {

    private static final Logger logger = LoggerFactory.getLogger(DeployitRemoteClient.class);

    private final DeployitCommunicator communicator;

    public DeployitRemoteClient(final DeployitCommunicator communicator) {
        this.communicator = communicator;
    }

    /**
     * Imports a package into XL Deploy bypassing resteasy.
     *
     * @param path Either URL or local path, or path on XL Deploy server
     * @param user Optional user
     * @param password Optional password
     * @return Configuration item as a result of import
     */
    public ConfigurationItem importPackage(String path, String user, String password) {
        try {
            URL url = new URL(path);
            if (url.getProtocol().equals("file")) {
                return importPackage(new File(url.getFile()));
            } else {
                return communicator.getProxies().getPackageService().fetch(new ImportFromUrlParams(url.toExternalForm(), user, password));
            }
        } catch (MalformedURLException e) {
            return importPackage(new File(path));
        }
    }

    public ConfigurationItem importPackage(String path) {
        return importPackage(path, null, null);
    }

    /**
     * Imports a package into XL Deploy bypassing resteasy. This method is preferably to use over {@link #importPackage(String)}
     * when you know that you are not going to deal with remote URLs.
     *
     * @param file either local file, or relative file path on XL Deploy server
     * @return Configuration item as a result of import
     */
    public ConfigurationItem importPackage(File file) {

        if (!file.isAbsolute() && !file.exists()) {
            return communicator.getProxies().getPackageService().importPackage(file.getPath().replace(File.separator, "/"));
        }

        logger.info("Importing dar file " + file.getAbsolutePath());

        try {
            return communicator.getProxies().getPackageService().upload(file.getName(), new StreamingImportingService.LocalFileUpload(file));
        } catch (FileNotFoundException e) {
            throw new IllegalArgumentException(format("The file %s does not exist at the given absolute path", file.toString()));
        }
    }

}
