/**
 * Copyright © 2014-2016 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.booter.remote;

import java.io.InputStream;
import java.lang.reflect.*;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import javax.ws.rs.BadRequestException;
import org.jboss.resteasy.client.jaxrs.*;
import org.jboss.resteasy.spi.ResteasyProviderFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.converters.DataHolder;
import com.thoughtworks.xstream.core.MapBackedDataHolder;

import com.xebialabs.deployit.booter.remote.client.ClientXStreamReaderWriter;
import com.xebialabs.deployit.booter.remote.resteasy.DeployitClientException;
import com.xebialabs.deployit.booter.remote.service.StreamingImportingService;
import com.xebialabs.deployit.core.api.ConfigurationService;
import com.xebialabs.deployit.engine.api.*;
import com.xebialabs.xltype.serialization.xstream.XStreamReaderWriter;

/**
 * The {@link com.xebialabs.deployit.booter.remote.Proxies} object is used to access services using RESTEasy.
 * The exposed services must never be cached, because on a re-login in {@link HttpClientHolder} the proxies will be invalidated.
 */
public class Proxies {
    private final Map<Class<?>, Object> registeredProxies = new HashMap<Class<?>, Object>();
    private final HttpClientHolder httpClientHolder;
    private final BooterConfig booterConfig;
    private final ResteasyClient client;

    /**
     * Use the second constructor
     */
    @Deprecated
    public Proxies(DeployitCommunicator communicator) {
        this(communicator.getHttpClientHolder(), communicator.getConfig());
    }

    public Proxies(HttpClientHolder httpClientHolder, BooterConfig config) {
        this.httpClientHolder = httpClientHolder;
        this.booterConfig = config;
        this.client = new ResteasyClientBuilder()
                .httpEngine(httpClientHolder.createClientHttpEngine())
                .providerFactory(new ResteasyProviderFactory(ResteasyProviderFactory.getInstance()))
                .register(new ClientXStreamReaderWriter(config))
                .build();
        init(config);
    }

    private void init(BooterConfig config) {
        for (Class<?> clazz : allProxies()) {
            registerProxy(config.getUrl(), clazz);
        }
    }

    public static Logger getLogger() {
        return logger;
    }

    public void registerProxy(String url, Class<?> clazz) {
        logger.debug("Registering Proxy: {}", clazz);
        registeredProxies.put(clazz, wrap(clazz, client.target(url).proxy(clazz)));
    }

    private Object wrap(final Class<?> clazz, final Object proxy) {
        InvocationHandler invocationHandler = Proxy.getInvocationHandler(proxy);
        return Proxy.newProxyInstance(Thread.currentThread().getContextClassLoader(), new Class[]{clazz}, new BadRequestExceptionHandler(invocationHandler, booterConfig));
    }

    @SuppressWarnings({"deprecation", "unchecked"})
    private static Iterable<Class<?>> allProxies() {
        ArrayList<Class<?>> classes = new ArrayList<>();
        Collections.addAll(classes, ServerService.class, MetadataService.class, RepositoryService.class,
                ControlService.class, DeploymentService.class, InspectionService.class, PackageService.class,
                ConfigurationService.class, PermissionService.class, RoleService.class, TaskService.class, TaskBlockService.class, UserService.class);
        return classes;
    }

    @SuppressWarnings("unchecked")
    public <T> T getProxyInstance(Class<T> clazz) {
        return (T) registeredProxies.get(clazz);
    }

    public ServerService getServerService() {
        return getProxyInstance(ServerService.class);
    }

    public MetadataService getMetadataService() {
        return getProxyInstance(MetadataService.class);
    }

    public RepositoryService getRepositoryService() {
        return getProxyInstance(RepositoryService.class);
    }

    public ControlService getControlService() {
        return getProxyInstance(ControlService.class);
    }

    public DeploymentService getDeploymentService() {
        return getProxyInstance(DeploymentService.class);
    }

    public InspectionService getInspectionService() {
        return getProxyInstance(InspectionService.class);
    }

    public PackageService getPackageService() {
        return new StreamingImportingService(httpClientHolder, booterConfig, this);
    }

    public PermissionService getPermissionService() {
        return getProxyInstance(PermissionService.class);
    }

    public RoleService getRoleService() {
        return getProxyInstance(RoleService.class);
    }

    @SuppressWarnings("deprecation")
    public TaskService getTaskService() {
        return getProxyInstance(TaskService.class);
    }

    public TaskBlockService getTaskBlockService() {
        return getProxyInstance(TaskBlockService.class);
    }

    public UserService getUserService() {
        return getProxyInstance(UserService.class);
    }

    private static final Logger logger = LoggerFactory.getLogger(Proxies.class);

    public static Proxies reinitialize(Proxies oldProxies, HttpClientHolder httpClientHolder, BooterConfig config) {

        Proxies newProxies = new Proxies(httpClientHolder, config);

        if (oldProxies == null) {
            return newProxies;
        }

        for (Class<?> proxyClass : oldProxies.registeredProxies.keySet()) {
            if (!newProxies.registeredProxies.containsKey(proxyClass)) {
                newProxies.registerProxy(config.getUrl(), proxyClass);
            }
        }


        return newProxies;
    }

    /**
     * Bad Request (HTTP 400) have special handling in the client.
     * <ul>
     *     <li>If the response has not body, an exception is thrown with some information passed from the server in headers (implemented in {@link com.xebialabs.deployit.booter.remote.resteasy.InternalServerErrorClientResponseInterceptor#tryDeployitException(javax.ws.rs.client.ClientResponseContext) Client reponse filter})</li>
     *     <li>If the response has a body, it is unmarshalled and if the object
     *          <ul>
     *              <li>is compatible with the return type of the invoked method, it is returned</li>
     *              <li>is not compatible, a {@link DeployitClientException} is thrown wrapping the object</li>
     *          </ul>
     *     </li>
     * </ul>
     */
    private static class BadRequestExceptionHandler implements InvocationHandler {
        private final InvocationHandler invocationHandler;
        private final BooterConfig booterConfig;

        public BadRequestExceptionHandler(final InvocationHandler invocationHandler, final BooterConfig booterConfig) {
            this.invocationHandler = invocationHandler;
            this.booterConfig = booterConfig;
        }

        @Override
        public Object invoke(final Object proxy, final Method method, final Object[] args) throws Throwable {
            try {
                return invocationHandler.invoke(proxy, method, args);
            } catch (BadRequestException bre) {
                logger.debug("Caught a BadRequestException with an entity");
                XStream xStream = XStreamReaderWriter.getConfiguredXStream();
                final DataHolder dataHolder = new MapBackedDataHolder();
                dataHolder.put("BOOTER_CONFIG", booterConfig.getKey());
                Object o = xStream.unmarshal(XStreamReaderWriter.HIERARCHICAL_STREAM_DRIVER.createReader(bre.getResponse().readEntity(InputStream.class)), null, dataHolder);
                if (method.getReturnType().isInstance(o)) {
                    return o;
                } else {
                    throw new DeployitClientException(o, bre.getResponse().getStatus());
                }
            }
        }
    }
}
