package com.xebialabs.deployit.booter.remote;

import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import org.jboss.resteasy.spi.ResteasyDeployment;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.Monitor;

import com.xebialabs.deployit.booter.remote.resteasy.InternalServerErrorClientResponseInterceptor;
import com.xebialabs.deployit.engine.xml.LocalDateStringConverter;
import com.xebialabs.deployit.engine.xml.XStreamReaderWriter;

import nl.javadude.scannit.Configuration;
import nl.javadude.scannit.Scannit;
import nl.javadude.scannit.scanner.MethodAnnotationScanner;
import nl.javadude.scannit.scanner.SubTypeScanner;
import nl.javadude.scannit.scanner.TypeAnnotationScanner;

import static com.google.common.collect.Sets.newHashSet;

public class RemoteBooter {

    private static final Monitor M = new Monitor();
    private static final AtomicBoolean BOOTED = new AtomicBoolean(false);
    private static final Map<BooterConfig, DeployitCommunicator> COMMUNICATORS = Maps.newConcurrentMap();
    private static ResteasyDeployment resteasyDeployment;

    public static DeployitCommunicator boot(BooterConfig config) {
        // Double checked locking FTW!
        if (!BOOTED.get()) {
            M.enter();
            try {
                doBoot(config);
            } finally {
                M.leave();
            }
        }
        return getCommunicator(config);
    }

    private static void doBoot(BooterConfig config) {
        if (!BOOTED.get()) {
            Scannit.boot(Configuration.config()
                    .scan("com.xebialabs")
                    .scan("ext.deployit") // Deployit Extensions
                    .with(new TypeAnnotationScanner(), new MethodAnnotationScanner(), new SubTypeScanner()));
            bootResteasy();

            DeployitCommunicator communicator = new DeployitCommunicator(config);
            COMMUNICATORS.put(config, communicator);
            RemoteDescriptorRegistry.boot(communicator);
            XStreamReaderWriter.registerConfigurationItemAliases();
            BOOTED.set(true);
        }
    }

    public static DeployitCommunicator getCommunicator(BooterConfig config) {
        if (!BOOTED.get()) {
            throw new IllegalStateException("Should first boot the RemoteBooter.");
        }

        if (!COMMUNICATORS.containsKey(config)) {
            M.enter();
            try {
                if (!COMMUNICATORS.containsKey(config)) {
                    COMMUNICATORS.put(config, new DeployitCommunicator(config));
                }
            } finally {
                M.leave();
            }
        }
        return COMMUNICATORS.get(config);
    }

    public static void shutdown() {
        M.enter();
        try {
            if (BOOTED.get()) {
                for (BooterConfig config : newHashSet(COMMUNICATORS.keySet())) {
                    COMMUNICATORS.get(config).shutdown();
                }
                resteasyDeployment.stop();
                BOOTED.set(false);
            }
        } finally {
            M.leave();
        }
    }

    private static void bootResteasy() {
        resteasyDeployment = new ResteasyDeployment();
        resteasyDeployment.getProviderClasses().add(XStreamReaderWriter.class.getName());
        resteasyDeployment.getProviderClasses().add(LocalDateStringConverter.class.getName());
        resteasyDeployment.start();
        resteasyDeployment.getProviderFactory().addClientErrorInterceptor(new InternalServerErrorClientResponseInterceptor());
    }

    private static final Logger logger = LoggerFactory.getLogger(RemoteBooter.class);

    static void remove(BooterConfig config) {
        COMMUNICATORS.remove(config);
    }
}
