package com.xebialabs.deployit.booter.remote.xml;

import java.util.Collection;
import java.util.List;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;

import com.xebialabs.deployit.engine.api.dto.ValidatedConfigurationItem;
import com.xebialabs.deployit.engine.xml.AbstractConfigurationItemConverter;
import com.xebialabs.deployit.engine.xml.XStreamProvider;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newHashSet;
import static com.google.common.collect.Sets.powerSet;

@XStreamProvider(tagName = "configuration-item", readable = ConfigurationItem.class)
public class ConfigurationItemConverter extends AbstractConfigurationItemConverter {

    @Override
    protected void writeCiProperty(Object value, PropertyDescriptor propertyDescriptor, HierarchicalStreamWriter writer) {
        //Remote a reference is an id, so write that immediately
        writer.addAttribute("ref", (String) value);
    }

    @Override
    protected void writeCollectionOfCiProperty(Object value, PropertyDescriptor propertyDescriptor, HierarchicalStreamWriter writer) {
        //Remote a collection of references is a collection<String>
        for (String s : (Collection<String>) value) {
            writer.startNode("ci");
            writer.addAttribute("ref", s);
            writer.endNode();
        }
    }

    @Override
    protected ConfigurationItem readProperties(HierarchicalStreamReader reader, Descriptor descriptor, ConfigurationItem configurationItem) {
        ConfigurationItem toBeReturned = configurationItem;
        while (reader.hasMoreChildren()) {
            reader.moveDown();
            if (reader.getNodeName().equals("validation-messages")) {
                toBeReturned = new ValidatedConfigurationItem(configurationItem);
                List<ValidationMessage> messages = newArrayList();
                while (reader.hasMoreChildren()) {
                    reader.moveDown();
                    messages.add(new ValidationMessage(reader.getAttribute("ci"), reader.getAttribute("property"), reader.getValue()));
                    reader.moveUp();
                }
                ((ValidatedConfigurationItem) toBeReturned).setValidations(messages);
            } else {
                readProperty(reader, descriptor, configurationItem);
            }
            reader.moveUp();
        }

        return toBeReturned;
    }

    @Override
    protected void readCiProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, HierarchicalStreamReader reader) {
        //Remote a CI is an id.
        propertyDescriptor.set(configurationItem, reader.getAttribute("ref"));
    }

    @Override
    protected void readCollectionOfCiProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, HierarchicalStreamReader reader) {
        //Remote a collection of CI references is a collection<String>
        Collection<String> c = newArrayList();

        while (reader.hasMoreChildren()) {
            reader.moveDown();
            c.add(reader.getAttribute("ref"));
            reader.moveUp();
        }

        propertyDescriptor.set(configurationItem, c);
    }

    @Override
    protected void readCollectionOfStringProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, HierarchicalStreamReader reader) {
        Collection<String> strings = Lists.<String>newArrayList();
        while (reader.hasMoreChildren()) {
            reader.moveDown();
            strings.add(reader.getValue());
            reader.moveUp();
        }
        propertyDescriptor.set(configurationItem, strings);
    }
}
