package com.xebialabs.deployit.booter.remote.client;

import java.io.File;
import java.io.FileNotFoundException;
import java.net.MalformedURLException;
import java.net.URL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.booter.remote.DeployitCommunicator;
import com.xebialabs.deployit.booter.remote.service.StreamingImportingService;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;

import static java.lang.String.format;

/**
 * A thin wrapper to close a gap between REST api functionality, provided out of the box by proxies and deployit clients needs.
 */
public class DeployitRemoteClient {


    private static final Logger logger = LoggerFactory.getLogger(DeployitRemoteClient.class);

    private final DeployitCommunicator communicator;

    public DeployitRemoteClient(final DeployitCommunicator communicator) {
        this.communicator = communicator;
    }

    /**
     * Imports a package into Deployit bypassing resteasy.
     *
     * @param path Either URL or local path, or path on Deployit server
     * @return Configuration item as a result of import
     */
    public ConfigurationItem importPackage(String path) {
        try {
            URL url = new URL(path);
            if (url.getProtocol().equals("file")) {
                return importPackage(new File(url.getFile()));
            } else {
                return communicator.getProxies().getPackageService().fetch(url.toExternalForm());
            }
        } catch (MalformedURLException e) {
            return importPackage(new File(path));
        }
    }

    /**
     * Imports a package into Deployit bypassing resteasy. This method is preferably to use over {@link #importPackage(String)}
     * when you know that you are not going to deal with remote URLs.
     *
     * @param file either local file, or relative file path on Deployit server
     * @return Configuration item as a result of import
     */
    public ConfigurationItem importPackage(File file) {

        if (!file.isAbsolute() && !file.exists()) {
            return communicator.getProxies().getPackageService().importPackage(file.getPath());
        }

        logger.info("Importing dar file " + file.getAbsolutePath());

        try {
            return communicator.getProxies().getPackageService().upload(file.getName(), new StreamingImportingService.LocalFileUpload(file));
        } catch (FileNotFoundException e) {
            throw new IllegalArgumentException(format("The file %s does not exist at the given absolute path", file.toString()));
        }
    }

}
