package com.xebialabs.deployit.booter.remote.xml;

import com.thoughtworks.xstream.converters.Converter;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;

import com.xebialabs.deployit.booter.remote.DeployitCommunicator;
import com.xebialabs.deployit.booter.remote.RemoteBooter;
import com.xebialabs.deployit.booter.remote.RemoteDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.MethodDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.xltype.serialization.xstream.Converters;
import com.xebialabs.xltype.serialization.xstream.XStreamProvider;

import static com.xebialabs.xltype.serialization.xstream.Converters.readList;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.Set;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newHashSet;

@XStreamProvider(tagName = "descriptor", readable = Descriptor.class)
public class DescriptorConverter implements Converter {
    @Override
    public void marshal(Object source, HierarchicalStreamWriter writer, MarshallingContext context) {
        throw new IllegalStateException("Cannot serialize Descriptors from remote-booter");
    }

    @Override
    public Object unmarshal(HierarchicalStreamReader reader, UnmarshallingContext context) {
        String booterConfigKey = (String) context.get("BOOTER_CONFIG");
        DeployitCommunicator communicator = RemoteBooter.getCommunicator(booterConfigKey);
        RemoteDescriptor descriptor = new RemoteDescriptor();
        setAttributes(reader, descriptor, communicator);
        while (reader.hasMoreChildren()) {
            reader.moveDown();
            if ("description".equals(reader.getNodeName())) {
                descriptor.setDescription(reader.getValue());
            } else if ("property-descriptors".equals(reader.getNodeName())) {
                List<PropertyDescriptor> list = readList(descriptor, PropertyDescriptor.class, reader, context);
                descriptor.setPropertyDescriptors(list);
            } else if ("control-tasks".equals(reader.getNodeName())) {
                List<MethodDescriptor> list = readList(descriptor, MethodDescriptor.class, reader, context);
                descriptor.setControlTasks(list);
            } else if ("interfaces".equals(reader.getNodeName())) {
                Set<Type> interfaces = newHashSet();
                while (reader.hasMoreChildren()) {
                    reader.moveDown();
                    interfaces.add(Type.valueOf(reader.getValue()));
                    reader.moveUp();
                }
                descriptor.setInterfaces(interfaces);
            } else if ("superTypes".equals(reader.getNodeName())) {
                List<Type> superTypes = newArrayList();
                while (reader.hasMoreChildren()) {
                    reader.moveDown();
                    superTypes.add(Type.valueOf(reader.getValue()));
                    reader.moveUp();
                }
                descriptor.setSuperTypes(superTypes);
            }
            reader.moveUp();
        }
        return descriptor;
    }

    private void setAttributes(HierarchicalStreamReader reader, RemoteDescriptor descriptor, DeployitCommunicator communicator) {
        descriptor.setType(communicator.getType(reader.getAttribute("type")));
        String deployableType = reader.getAttribute("deployableType");
        if (deployableType != null) {
            descriptor.setDeployableType(communicator.getType(deployableType));
        }

        String containerType = reader.getAttribute("containerType");
        if (containerType != null) {
            descriptor.setContainerType(communicator.getType(containerType));
        }

        String virtual = reader.getAttribute("virtual");
        if ("true".equalsIgnoreCase(virtual)) {
            descriptor.setVirtual();
        }

        String inspectable = reader.getAttribute("inspectable");
        if ("true".equalsIgnoreCase(inspectable)) {
            descriptor.setInspectable();
        }

        descriptor.setRoot(readRoot(reader));
    }

    private Metadata.ConfigurationItemRoot readRoot(HierarchicalStreamReader reader) {
        String rootNodeName = reader.getAttribute("root");
        if (rootNodeName == null) {
            return Metadata.ConfigurationItemRoot.NESTED;
        }
        for (Metadata.ConfigurationItemRoot configurationItemRoot : Metadata.ConfigurationItemRoot.values()) {
            if (rootNodeName.equals(configurationItemRoot.getRootNodeName())) {
                return configurationItemRoot;
            }
        }
        return null;
    }

    @Override
    public boolean canConvert(Class type) {
        return type.equals(Descriptor.class);
    }

    private static final Logger logger = LoggerFactory.getLogger(DescriptorConverter.class);
}
