package com.xebialabs.deployit.booter.remote.xml;

import java.util.List;
import com.google.common.base.Predicate;
import com.google.common.collect.Lists;
import com.thoughtworks.xstream.converters.DataHolder;

import com.xebialabs.deployit.booter.remote.DeployitCommunicator;
import com.xebialabs.deployit.booter.remote.RemoteBooter;
import com.xebialabs.deployit.booter.remote.xml.adapter.CiPropertyAdapter;
import com.xebialabs.deployit.booter.remote.xml.adapter.OrchestratorPropertyAdapter;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.xltype.serialization.CiReader;
import com.xebialabs.xltype.serialization.CiWriter;
import com.xebialabs.xltype.serialization.ConfigurationItemConverter;
import com.xebialabs.xltype.serialization.xstream.XStreamCiConverter;
import com.xebialabs.xltype.serialization.xstream.XStreamProvider;

import static com.google.common.collect.Iterables.find;

@XStreamProvider(tagName = "configuration-item", readable = ConfigurationItem.class)
public class RemoteXStreamCiConverter extends XStreamCiConverter {

    private List<CiPropertyAdapter> propertyAdapters;

    public RemoteXStreamCiConverter() {
        this.propertyAdapters = Lists.<CiPropertyAdapter>newArrayList(new OrchestratorPropertyAdapter());
    }

    @Override
    protected ConfigurationItemConverter createConverter(DataHolder context) {
        return new RemoteCiConverter(context, propertyAdapters);
    }

    public void setPropertyAdapters(final List<CiPropertyAdapter> propertyAdapters) {
        this.propertyAdapters = propertyAdapters;
    }

    private static class RemoteCiConverter extends ConfigurationItemConverter {

        private DataHolder context;
        private List<CiPropertyAdapter> propertyAdapters;

        public RemoteCiConverter(DataHolder context) {
            this(context, Lists.<CiPropertyAdapter>newArrayListWithCapacity(0));
        }

        public RemoteCiConverter(DataHolder context, List<CiPropertyAdapter> propertyAdapters) {
            this.context = context;
            this.propertyAdapters = propertyAdapters;
            setReadValidationMessages(true);
        }

        @Override
        public Type type(String typeName) {
            DeployitCommunicator communicator = getCommunicator();
            return communicator.getType(typeName);
        }

        private DeployitCommunicator getCommunicator() {
            String booterConfigKey = (String) context.get("BOOTER_CONFIG");
            return RemoteBooter.getCommunicator(booterConfigKey);
        }

        @Override
        protected void readCiProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, CiReader reader) {
            // Store reference directly on the CI as String.
            propertyDescriptor.set(configurationItem, reader.getCiReference());
        }

        @Override
        protected void readCollectionOfCiProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, CiReader reader) {
            // Store references directly on the CI as String.
            // For Python's sake, a SET_OF_CI is also stored as a list.
            List<String> references = reader.getCiReferences();
            propertyDescriptor.set(configurationItem, references);
        }

        @Override
        protected void readCollectionOfStringProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, CiReader reader) {
            // For Python's sake, a SET_OF_STRING is also stored as a list.
            List<String> strings = reader.getStringValues();
            propertyDescriptor.set(configurationItem, strings);
        }

        @Override
        protected void writeProperty(final ConfigurationItem configurationItem, final PropertyDescriptor propertyDescriptor, final CiWriter writer, final int ciRefsFromLevel) {
            final CiPropertyAdapter adapter = getAdapter(context, configurationItem, propertyDescriptor);

            if(adapter != null) {
                adapter.writeProperty(configurationItem, propertyDescriptor, writer, ciRefsFromLevel);
            } else {
                super.writeProperty(configurationItem, propertyDescriptor, writer, ciRefsFromLevel);
            }
        }

        @Override
        protected void readProperty(final CiReader reader, final Descriptor descriptor, final ConfigurationItem configurationItem) {
            PropertyDescriptor propertyDescriptor = descriptor.getPropertyDescriptor(reader.getCurrentPropertyName());
            final CiPropertyAdapter adapter = getAdapter(context, configurationItem, propertyDescriptor);
            if (adapter != null) {
                adapter.readProperty(reader, propertyDescriptor, configurationItem);
            } else {
                super.readProperty(reader, descriptor, configurationItem);
            }
        }

        private CiPropertyAdapter getAdapter(final DataHolder context, final ConfigurationItem configurationItem, final PropertyDescriptor propertyDescriptor) {
            return find(propertyAdapters, new Predicate<CiPropertyAdapter>() {
                @Override
                public boolean apply(final CiPropertyAdapter input) {
                    return input.isApplicable(context, configurationItem, propertyDescriptor);
                }
            }, null);
        }
    }
}
