package com.xebialabs.deployit.engine.replacer;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Sets;
import com.google.common.io.ByteSink;
import com.google.common.io.Files;
import com.xebialabs.deployit.engine.unicode.BOM;
import com.xebialabs.deployit.engine.unicode.DetectBOM;
import com.xebialabs.deployit.plugin.api.udm.artifact.ArchiveArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.FolderArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.RuntimeIOException;
import com.xebialabs.overthere.local.LocalFile;
import de.schlichtherle.truezip.file.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.nio.charset.Charset;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.google.common.base.Preconditions.*;
import static com.google.common.base.Strings.emptyToNull;
import static com.xebialabs.deployit.plugin.api.udm.base.BaseDeployableArtifact.SCAN_PLACEHOLDERS_PROPERTY_NAME;
import static com.xebialabs.overthere.util.OverthereUtils.closeQuietly;
import static java.lang.Math.abs;
import static java.lang.String.format;
import static java.util.regex.Pattern.CASE_INSENSITIVE;
import static java.util.regex.Pattern.COMMENTS;

public class Placeholders {

    public static final String DEFAULT_DELIMS = "{{ }}";
    public static LoadingCache<String, Pattern> patternMap = CacheBuilder.newBuilder().build(new CacheLoader<String, Pattern>() {
        @Override
        public Pattern load(final String key) throws Exception {
            return Pattern.compile(key, COMMENTS | CASE_INSENSITIVE);
        }
    });

    public static void scanPlaceholders(SourceArtifact artifact, PlaceholderScanner scanner) {
        OverthereFile artifactFile = artifact.getFile();
        checkArgument(artifactFile != null, artifact + " has no file");
        checkArgument(artifactFile instanceof LocalFile, "Cannot scan for placeholders in [%s] because its file is not a LocalFile but a [%s]", artifact, artifactFile.getClass().getName());

        artifact.setPlaceholders(Sets.<String> newTreeSet());
        if (!shouldScanPlaceholders(artifact)) {
            logger.debug("Artifact [{}] has disabled placeholder scanning", artifact);
            return;
        }
        TFile from = asTfile(artifactFile);
        try {
            doScanPlaceholders(artifact, from, scanner, true, artifactFile.getPath());
        } finally {
            umountQuietly(from);
        }
    }

    public static boolean shouldScanPlaceholders(SourceArtifact artifact) {
        return !artifact.hasProperty(SCAN_PLACEHOLDERS_PROPERTY_NAME) || (Boolean) artifact.getProperty(SCAN_PLACEHOLDERS_PROPERTY_NAME);
    }

    private static void doScanPlaceholders(SourceArtifact artifact, TFile from, PlaceholderScanner scanner, boolean isRoot, String basePath) {
        if (shouldExcludeFile(from, artifact, isRoot)) {
            return;
        }

        logger.trace("File [{}] is a [directory, archive, file]: [{}, {}, {}]", from, from.isDirectory(), from.isArchive(), from.isFile());

        if (from.isDirectory()) {
            try {
                for (TFile f : from.listFiles(TArchiveDetector.ALL)) {
                    doScanPlaceholders(artifact, f, scanner, false, basePath);
                }
            } finally {
                umountQuietly(from);
            }
        } else if (isTextFile(from.getName(), artifact.getTextFileNamesRegex())) {
            artifact.getPlaceholders().addAll(readPlaceholders(artifact, from, scanner, basePath));
        }
    }

    private static Set<String> readPlaceholders(SourceArtifact artifact, final TFile from, final PlaceholderScanner scanner, String basePath) {
        try {
            Charset charset = getCharset(artifact, from, basePath);

            Reader in;
            if (charset == null) {
                logger.debug("Replacing placeholders in [{}] using default charset", from);
                in = new TFileReader(from);
            } else {
                logger.debug("Replacing placeholders in [{}] using charset [{}]", from, charset);
                in = new TFileReader(from, charset.newDecoder());
            }

            try {
                if (artifact.hasProperty("delimiters")) {
                    return scanner.scan(in, artifact.<String> getProperty("delimiters"));
                } else {
                    return scanner.scan(in);
                }
            } finally {
                closeQuietly(in);
            }
        } catch (IOException exc) {
            throw new RuntimeIOException(format("Cannot scan placeholders in [%s]", from), exc);
        } catch (RuntimeException exc) {
            throw new RuntimeException(format("Cannot scan placeholders in [%s]", from), exc);
        }
    }

    public static void replacePlaceholders(DerivedArtifact<? extends SourceArtifact> derivedArtifact, PlaceholderReplacer replacer) {
        if (derivedArtifact.getSourceArtifact() == null) {
            derivedArtifact.setFile(null);
        } else {
            OverthereFile fromFile = derivedArtifact.getSourceArtifact().getFile();
            checkArgument(fromFile != null, "[%s] has no file", derivedArtifact.getSourceArtifact());
            checkArgument(fromFile instanceof LocalFile, "Cannot replace placeholders in [%s] because its file is not a LocalFile but a [%s]", derivedArtifact.getSourceArtifact(), fromFile.getClass().getName());

            TFile from = null;
            try {
                from = getTFileWithCorrectDirectoryDetection(fromFile);
                boolean isBinaryFile = from.isFile() && !isTextFile(fromFile.getName(), getTextFileNamesRegex(derivedArtifact));
                if (derivedArtifact.getPlaceholders().isEmpty() || isBinaryFile) {
                    derivedArtifact.setFile(fromFile);
                } else {
                    OverthereFile derivedFile = getOutputFile(derivedArtifact);
                    try {
                        fromFile.copyTo(derivedFile);
                        TFile to = getTFileWithCorrectDirectoryDetection(derivedFile);
                        try {
                            doReplacePlaceholders(derivedArtifact, to, replacer, true, derivedFile.getPath());
                            saveArchive(to);
                        } finally {
                            umountQuietly(to);
                        }
                        derivedArtifact.setFile(derivedFile);
                    } finally {
                        checkState(derivedFile.getParentFile().listFiles().size() == 1, "Should only be one file in the deployed dir, was %s", derivedFile
                            .getParentFile().listFiles());
                    }
                }
            } finally {
                umountQuietly(from);
            }
        }
    }

    private static TFile getTFileWithCorrectDirectoryDetection(final OverthereFile derivedFile) {
        final TFile to;// Disable archive detection for directories (useful when the directory is named
                       // PetClinic.war)...
        // Also disable archive detection for the parent directory of the artifact (useful when the artifact is named
        // PetClinic.war)
        if (derivedFile.isDirectory()) {
            to = new TFile(derivedFile.getPath(), TArchiveDetector.NULL);
        } else {
            String parentPath = derivedFile.getParentFile().getPath();
            to = new TFile(new TFile(parentPath, TArchiveDetector.NULL), derivedFile.getName(), TArchiveDetector.ALL);
        }
        return to;
    }

    private static void doReplacePlaceholders(DerivedArtifact<? extends SourceArtifact> derivedArtifact, TFile to, PlaceholderReplacer replacer, boolean isRoot, String basePath) {
        boolean include = !shouldExcludeFile(to, derivedArtifact.getSourceArtifact(), isRoot);

        if (include && to.isDirectory()) {
            try {
                for (TFile t : to.listFiles(TArchiveDetector.ALL)) {
//                    t = asTfile(t);
                    doReplacePlaceholders(derivedArtifact, t, replacer, false, basePath);
                }
            } finally {
                umountQuietly(to);
            }
        } else if (include && isTextFile(to.getName(), getTextFileNamesRegex(derivedArtifact))) {
            replace(to, replacer, derivedArtifact, basePath);
        }
    }

    /**
     * Replacement needs a temporary file to write to.
     */
    private static void replace(final TFile toBeReplaced, final PlaceholderReplacer replacer, DerivedArtifact<? extends SourceArtifact> artifact, String basePath) {
        try {
            Charset charset = getCharset(artifact.getSourceArtifact(), toBeReplaced, basePath);

            File tempFile = File.createTempFile(toBeReplaced.getName(), ".tmp");
            try {
                Reader reader;
                Writer writer;
                if (charset == null) {
                    logger.debug("Replacing placeholders in [{}] using default charset", toBeReplaced);
                    reader = new TFileReader(toBeReplaced);
                    writer = new OutputStreamWriter(new FileOutputStream(tempFile));
                } else {
                    logger.debug("Replacing placeholders in [{}] using charset [{}]", toBeReplaced, charset);
                    reader = new TFileReader(toBeReplaced, charset.newDecoder());
                    writer = new OutputStreamWriter(new FileOutputStream(tempFile), charset.newEncoder());
                }

                try {
                    if (artifact.getSourceArtifact().hasProperty("delimiters")) {
                        replacer.replace(reader, writer, artifact.getPlaceholders(), artifact.getSourceArtifact().<String> getProperty("delimiters"));
                    } else {
                        replacer.replace(reader, writer, artifact.getPlaceholders());
                    }
                } finally {
                    closeQuietly(reader);
                    closeQuietly(writer);
                }

                Files.asByteSource(tempFile).copyTo(new ByteSink() {
                    @Override
                    public OutputStream openStream() throws IOException {
                        return new TFileOutputStream(toBeReplaced);
                    }
                });
            } finally {
                if (!tempFile.delete()) {
                    logger.warn("Cannot delete temporary file [{}]", tempFile);
                }
            }
        } catch (IOException | RuntimeException exc) {
            throw new RuntimeIOException(format("Cannot replace placeholders in [%s]", toBeReplaced), exc);
        }
    }

    private static OverthereFile getOutputFile(DerivedArtifact<? extends SourceArtifact> derivedArtifact) {
        OverthereFile sourceFile = derivedArtifact.getSourceArtifact().getFile();
        OverthereFile workDir = sourceFile.getParentFile();

        Random r = new Random();
        String baseName = derivedArtifact.getName();
        for (;;) {
            String name = baseName + abs(r.nextInt());
            OverthereFile deployedArtifactDir = workDir.getFile(name);

            if (!deployedArtifactDir.exists()) {
                deployedArtifactDir.mkdir();
                return deployedArtifactDir.getFile(sourceFile.getName());
            }
        }
    }

    private static File saveArchive(TFile outputArchive) {
        if (outputArchive.isArchive() && outputArchive.getEnclArchive() == null && outputArchive.isDirectory()) {
            try {
                TVFS.umount(outputArchive);
            } catch (IOException exc) {
                throw new RuntimeIOException(format("Cannot write archive [%s]", outputArchive), exc);
            }
        }

        // Return a regular java.io.File pointing to the file, folder or archive just written
        return new File(outputArchive.getPath());
    }

    private static boolean shouldExcludeFile(final TFile f, final SourceArtifact artifact, boolean isRoot) {

        if (isRoot && artifact instanceof FolderArtifact) {
            return false;
        }

        if (emptyToNull(artifact.getExcludeFileNamesRegex()) == null) {
            return false;
        }

        Pattern excludeFileNamesPattern = patternMap.getUnchecked(artifact.getExcludeFileNamesRegex());
        Matcher excludeFileNamesMatcher = excludeFileNamesPattern.matcher(f.getPath());
        boolean exclude = excludeFileNamesMatcher.matches();
        if (exclude) {
            logger.debug("Excluding file [{}] from scanning", f);
        }
        return exclude;
    }

    private static boolean isTextFile(final String name, final String textFileNamesRegex) {
        checkNotNull(textFileNamesRegex, "Regex is null");

        Pattern textFileNamesPattern = patternMap.getUnchecked(textFileNamesRegex);
        Matcher textFileNamesMatcher = textFileNamesPattern.matcher(name);
        boolean isTextFile = textFileNamesMatcher.matches();
        logger.debug("Determined [{}] to be a {} file", name, isTextFile ? "text" : "binary");

        return isTextFile;
    }

    private static String getTextFileNamesRegex(DerivedArtifact<? extends SourceArtifact> derivedArtifact) {
        return derivedArtifact.getSourceArtifact().getTextFileNamesRegex();
    }

    private static TFile asTfile(OverthereFile file) {
        File from = ((LocalFile) file).getFile();
        if (from.isDirectory()) {
            return new TFile(from, TArchiveDetector.NULL);
        }
        return new TFile(from);
    }

    static void umountQuietly(TFile file) {
        if (file != null && file.isArchive() && file.getEnclArchive() == null) {
            try {
                TVFS.umount(file);
            } catch (Exception e) {
                logger.error("Cannot umount [{}], ignoring exception.", file);
                logger.debug("Exception while umounting was: ", e);
            }
        }
    }

    /**
     * Will return 'null' if the default charset of the JVM needs to be used.
     * TODO return UTF-8 instead of 'null' for 4.0
     *
     * @return charset
     */
    private static Charset getCharset(SourceArtifact artifact, TFile file, String basePath) throws IOException {
        String relativePath = getRelativePath(artifact, file, basePath);
        logger.trace("Relative path = {}", relativePath);
        Map<String,String> fileEncodings = artifact.getFileEncodings();
        for (Map.Entry<String, String> regexToCharset : fileEncodings.entrySet()) {
            if (relativePath.matches(regexToCharset.getKey())) {
                logger.debug("Relative path [{}] matched regex [{}], using charset [{}]", relativePath, regexToCharset.getKey(), regexToCharset.getValue());
                return Charset.forName(regexToCharset.getValue());
            }
        }

        BOM detect = DetectBOM.detect(file);
        return detect.getCharset();
    }

    private static String getRelativePath(SourceArtifact artifact, TFile file, String basePath) {
        if (artifact instanceof FolderArtifact || artifact instanceof ArchiveArtifact) {
            return file.getPath().substring(basePath.length() + 1);
        } else {
            return file.getName();
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(Placeholders.class);

}
