package com.xebialabs.deployit.engine.tasker;

import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;

import org.joda.time.DateTime;

import com.google.common.collect.Lists;

import com.xebialabs.deployit.engine.api.execution.StepState;
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState;
import com.xebialabs.deployit.engine.api.execution.TaskWithSteps;

import static com.google.common.base.Preconditions.checkArgument;

public class Task implements Serializable, TaskWithSteps {
    private List<TaskStep> steps;
    private final String id;
    private final String description;
    private DateTime startDate;
    private DateTime completionDate;
    private volatile TaskExecutionState state;
    private AtomicInteger failureCount;
    private AtomicInteger stepNr;
    private TaskExecutionContext context;
    private Map<String, String> metadata;
    private String owner;

    Task(List<TaskStep> steps, TaskSpecification spec) {
        this.steps = steps;
        this.description = spec.getDescription();
        this.metadata = spec.getMetadata();
        this.owner = spec.getOwner();
        this.id = UUID.randomUUID().toString();
        this.state = TaskExecutionState.PENDING;
        this.failureCount = new AtomicInteger(0);
        this.stepNr = new AtomicInteger(0);
        this.context = new TaskExecutionContext(spec.getListeners(), spec.getInspectionContext());
        if (spec.getInspectionContext() != null) {
            spec.getInspectionContext().registerTask(this);
        }
    }

    @Override
    public List<StepState> getSteps() {
        return Lists.<StepState>newArrayList(steps);
    }

    public List<TaskStep> getTaskSteps() {
        return steps;
    }

    @Override
    public TaskExecutionState getState() {
        return state;
    }


    @Override
    public String getId() {
        return id;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public DateTime getStartDate() {
        return startDate;
    }

    @Override
    public DateTime getCompletionDate() {
        return completionDate;
    }

    @Override
    public int getNrSteps() {
        return steps.size();
    }

    public TaskStep getCurrentStep() {
        if (stepNr.intValue() == 0) {
            return null;
        }
        return getStep(stepNr.intValue());
    }

    public TaskStep getStep(int stepNr) {
        return steps.get(stepNrToIndex(stepNr));
    }

    int stepNrToIndex(int stepNr) {
        checkArgument(stepNr > 0 && stepNr <= steps.size(), "Not a valid step number [%s]", stepNr);
        return stepNr - 1;
    }

    @Override
    public int getCurrentStepNr() {
        return stepNr.intValue();
    }

    @Override
    public Map<String, String> getMetadata() {
        return metadata;
    }

    @Override
    public int getFailureCount() {
        return failureCount.intValue();
    }

    public TaskExecutionContext getContext() {
        return context;
    }

    public String getOwner() {
        return owner;
    }

    public void setOwner(String owner) {
        this.owner = owner;
    }

    void recordFailure() {
        failureCount.incrementAndGet();
    }

    void recordStart() {
        if (this.startDate == null) {
            this.startDate = new DateTime();
        }
    }

    void recordCompletion() {
        this.completionDate = new DateTime();
    }

    boolean recordNextStep() {
        if (stepNr.get() < steps.size()) {
            stepNr.incrementAndGet();
            return true;
        }
        return false;
    }

    void setState(TaskExecutionState state) {
        this.state = state;
    }

    @Override
    public String toString() {
        return id + " - " + description;
    }
}
