package com.xebialabs.deployit.engine.tasker;

import java.util.List;
import java.util.concurrent.FutureTask;

import nl.javadude.t2bus.BusException;
import nl.javadude.t2bus.event.strategy.ThrowingEventHandlerStrategy;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import com.xebialabs.deployit.engine.api.execution.StepExecutionState;
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState;
import com.xebialabs.deployit.engine.tasker.step.PauseStep;
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.services.Repository;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static com.xebialabs.deployit.engine.api.execution.StepExecutionState.DONE;
import static com.xebialabs.deployit.engine.api.execution.StepExecutionState.FAILED;
import static com.xebialabs.deployit.engine.api.execution.StepExecutionState.PAUSED;
import static com.xebialabs.deployit.engine.api.execution.StepExecutionState.SKIP;
import static com.xebialabs.deployit.engine.api.execution.StepExecutionState.SKIPPED;
import static com.xebialabs.deployit.engine.api.execution.TaskExecutionState.CANCELLED;
import static com.xebialabs.deployit.engine.api.execution.TaskExecutionState.EXECUTED;
import static com.xebialabs.deployit.engine.api.execution.TaskExecutionState.EXECUTING;
import static com.xebialabs.deployit.engine.api.execution.TaskExecutionState.QUEUED;
import static com.xebialabs.deployit.engine.api.execution.TaskExecutionState.STOPPED;
import static com.xebialabs.deployit.engine.tasker.TaskEvent.taskEvent;

public class TaskRunner implements Runnable {
    private static final String MDC_KEY_TASK_ID = "taskId";
    private static final String MDC_KEY_STEP_DESCRIPTION = "stepDescription";
    private static final String MDC_KEY_USERNAME = "username";

    private volatile Task task;
    private volatile boolean stopRequested;

    private volatile FutureTask<Object> threadHandle;

    private TaskExecutionContext context;
    private ExecutionContext executionContext;

    public TaskRunner(Task task, Repository repository) {
        this.task = task;
        this.context = task.getContext();
        this.context.init(new EventBusAdapter(), repository);
    }

    @Override
    public void run() {
        MDC.put(MDC_KEY_TASK_ID, task.getId());
        MDC.put(MDC_KEY_USERNAME, task.getOwner());
        task.recordStart();

        try {
            stopRequested = false;
            checkState(task.getState() == QUEUED, "Task should be queued");
            checkState(threadHandle != null, "task should have a wrapper set!");

            try {
                // Set to next step so that we have a log when stuff goes wrong.
                if (task.getCurrentStep() == null) {
                    nextStep();
                } else {
                    executionContext = task.getCurrentStep().getContext(context);
                }
                setTaskState(EXECUTING);
                final boolean result = executeSteps();
                if (!result && isAborted()) {
                    logger.debug("Task aborted.");
                    task.recordFailure();
                    setTaskState(STOPPED);
                } else if (!result && stopRequested) {
                    logger.debug("Stopped the task.");
                    task.recordFailure();
                    setTaskState(STOPPED);
                } else if (!result) {
                    logger.debug("Step failed: {}", task.getCurrentStep());
                    task.recordFailure();
                    setTaskState(STOPPED);
                } else {
                    logger.debug("Executed all steps of {}", task);
                    task.recordCompletion();
                    setTaskState(TaskExecutionState.EXECUTED);
                }
            } catch (RuntimeException re) {
                executionContext.logError("Task failed.", re);
                throw re;
            } finally {
                // I'm done, don't need my threadHandle anymore!
                threadHandle = null;
                // Clear the abort flag
                Thread.interrupted();
            }
        } finally {
            MDC.remove(MDC_KEY_TASK_ID);
            MDC.remove(MDC_KEY_USERNAME);
        }
    }

    public void queue() {
        checkState(canBeQueued(), "Can only queue a task that is PENDING, state is now [%s]", task.getState());
        setTaskState(QUEUED);
    }

    public void abort() {
        if (isExecuting()) {
            threadHandle.cancel(true);
        } else {
            logger.info("Cannot abort task [{}] which is not currently executing.", task.getId());
        }
    }

    public void cancel() {
        checkArgument(canBeQueued() || isReadyForExecution(), "Can only cancel a task that is PENDING, QUEUED or STOPPED; state is now [%s]", task.getState());
        task.recordCompletion();
        setTaskState(CANCELLED);
    }

    public void stop() {
        stopRequested = true;
    }

    public void archive() {
        checkState(task.getState() == EXECUTED, "Task can only be archived if it has been executed.");
        setTaskState(TaskExecutionState.DONE);
    }

    public void skip(int stepNr) {
        checkState(canBeQueued(), "Task [%s] should be PENDING or STOPPED, [%s]", task.getId(), task.getState());
        TaskStep step = task.getStep(stepNr);
        checkArgument(step.canSkip(), "Step [%s] cannot be skipped.", stepNr);
        setStepState(step, SKIP);
    }

    public void unskip(Integer stepNr) {
        checkState(canBeQueued(), "Task [%s] should be PENDING or STOPPED, [%s]", task.getId(), task.getState());
        TaskStep step = task.getStep(stepNr);
        checkArgument(step.isMarkedForSkip(), "Step [%s] cannot be un-skipped", stepNr);
        setStepState(step, StepExecutionState.PENDING);
    }

    public void moveStep(int stepNr, int newPosition) {
        checkState(task.getState() == TaskExecutionState.PENDING && task.getCurrentStepNr() == 0, "Task should not have run when moving steps, [%s]", task.getId());
        List<TaskStep> taskSteps = task.getTaskSteps();
        int index = task.stepNrToIndex(stepNr);
        int index1 = task.stepNrToIndex(newPosition);
        TaskStep moved = taskSteps.remove(index);
        taskSteps.add(index1, moved);
    }

    public void addPause(int position) {
        checkState(canBeQueued(), "Task [%s] should be PENDING or STOPPED, [%s]", task.getId(), task.getState());
        checkState(task.getCurrentStepNr() < position, "Can only add pause steps after the current execution position (currently: [%s], requested: [%s])", task.getCurrentStepNr(), position);
        task.getTaskSteps().add(task.stepNrToIndex(position), new TaskStep(new PauseStep()));
    }

    void setThreadHandle(FutureTask<Object> future) {
        this.threadHandle = future;
    }

    public Task getTask() {
        return task;
    }

    private boolean executeSteps() {
        while (hasMoreSteps()) {
            TaskStep step = task.getCurrentStep();
            executeStep(step, task.getContext());
            if (step.getState() == FAILED || step.getState() == PAUSED) {
                return false;
            }

            if (!nextStep()) {
                break;
            }

            if (stopRequested) {
                return false;
            }
        }

        return true;
    }

    private boolean nextStep() {
        boolean hasNextStep = task.recordNextStep();
        if (hasNextStep) {
            executionContext = task.getCurrentStep().getContext(context);
        }
        return hasNextStep;
    }

    private void executeStep(TaskStep step, TaskExecutionContext taskContext) {
        Step implementation = step.getImplementation();
        checkState(step.getState() != StepExecutionState.EXECUTING, "The step [%s] is still executing, cannot run again.", step);

        MDC.put(MDC_KEY_STEP_DESCRIPTION, implementation.getDescription());
        try {
            if (step.getState().isFinal()) {
                logger.debug("Will not execute: {} with description: {} because it has state: {}", new Object[]{implementation, implementation.getDescription(), step.getState()});
                return;
            }

            step.recordStart();

            if (step.isMarkedForSkip()) {
                setStepState(step, SKIPPED);
                step.recordCompletion();
                return;
            }

            setStepState(step, StepExecutionState.EXECUTING);
            step.clearLog();
            logger.info("Started {}", this);
            try {
                final StepExitCode result = implementation.execute(executionContext);
                switch (result) {
                    case SUCCESS:
                        setStepState(step, DONE);
                        break;
                    case FAIL:
                        setStepState(step, FAILED);
                        break;
                    case PAUSE:
                        setStepState(step, PAUSED);
                        break;
                }
            } catch (Exception exc) {
                executionContext.logError("Step failed", exc);
                setStepState(step, FAILED);
            } catch (Throwable t) {
                // Any non-exception throwable is so severe we abort the task!
                executionContext.logError("Step failed badly, aborting!", t);
                Thread.currentThread().interrupt();
                setStepState(step, FAILED);
            } finally {
                TaskStep.logger.info(step.getState().name());
            }
            step.recordCompletion();
        } finally {
            MDC.remove(MDC_KEY_STEP_DESCRIPTION);
        }
    }

    private void setTaskState(TaskExecutionState newState) {
        TaskEvent e = taskEvent(task, newState);
        task.setState(newState);
        try {
            context.publish(e, new ThrowingEventHandlerStrategy());
        } catch (BusException be) {
            if (!stopRequested) {
                stop();
                setTaskState(STOPPED);
            }
            throw be;
        }
    }

    private void setStepState(TaskStep step, StepExecutionState newState) {
        StepEvent e = new StepEvent(step.getState(), newState, task, step);
        step.setState(newState);
        try {
            context.publish(e, new ThrowingEventHandlerStrategy());
        } catch (BusException be) {
            if (!stopRequested) {
                stop();
                setTaskState(STOPPED);
            }
            throw be;
        }
    }

    private boolean hasMoreSteps() {
        return task.getNrSteps() >= task.getCurrentStepNr();
    }

    private boolean isAborted() {
        return threadHandle.isCancelled() || Thread.currentThread().isInterrupted();
    }

    private boolean isReadyForExecution() {
        return task.getState() == QUEUED;
    }

    boolean isExecuting() {
        return task.getState() == EXECUTING;
    }

    private boolean canBeQueued() {
        return task.getState() == TaskExecutionState.PENDING || task.getState() == TaskExecutionState.STOPPED;
    }

    FutureTask<Object> getThreadHandle() {
        return threadHandle;
    }

    @Override
    public String toString() {
        return task.toString();
    }

    private static final Logger logger = LoggerFactory.getLogger(TaskRunner.class);

}
