package com.xebialabs.deployit.test.repository;

import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.platform.test.TestUtils.newInstance;

public class InMemoryRepository implements Repository {

    public static final AtomicReference<InMemoryRepository> REFERENCE = new AtomicReference<InMemoryRepository>(new InMemoryRepository());

    private Map<String, ConfigurationItem> store = Maps.newHashMap();

    public void initializeWithFakeRootFolders() {
        create(newInstance("test.Root", "Configuration"));
        create(newInstance("test.Root", "Infrastructure"));
        create(newInstance("test.Root", "Applications"));
        create(newInstance("test.Root", "Environments"));
    }

    @Override
    public boolean exists(String id) {
        return store.containsKey(id);
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T extends ConfigurationItem> T read(String id) {
        if (!exists(id)) {
            throw new IllegalArgumentException("Repository does not contain item with id " + id);
        }

        return (T) store.get(id);
    }

    @Override

    public <T extends ConfigurationItem> void create(T... entities) {
        checkThatEntitiesDoNotExist(entities);
        storeEntities(entities);
    }

    @Override
    public <T extends ConfigurationItem> void update(T... entities) {
        storeEntities(entities);
    }

    @Override
    public <T extends ConfigurationItem> void createOrUpdate(T... entities) {
        storeEntities(entities);
    }

    @Override
    public void delete(String... ids) {
        for (String id : ids) {
            store.remove(id);
        }
    }

    @Override
    public void move(String id, String newId) {
        ConfigurationItem ci = store.remove(id);
        ci.setId(newId);
        store.put(newId, ci);
    }

    @Override
    public void rename(String id, String newName) {
        int i = id.lastIndexOf("/");
        String newId = newName;
        if (i > -1) {
            newId = id.substring(0, i + 1) + newName;
        }
        move(id, newId);
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T extends ConfigurationItem> List<T> search(Type type) {
        List<T> entities = newArrayList();
        for (ConfigurationItem configurationItem : store.values()) {
            if (type == null || configurationItem.getType().instanceOf(type)) {
                entities.add((T) configurationItem);
            }
        }
        return entities;
    }

    @Override
    public <T extends ConfigurationItem> List<T> search(Type type, final String parent) {
        return newArrayList(Iterables.<T>filter(this.<T>search(type), new Predicate<ConfigurationItem>() {
            public boolean apply(ConfigurationItem input) {
                return input.getId().startsWith(parent);
            }
        }));
    }

    private void checkThatEntitiesDoNotExist(final ConfigurationItem[] entities) {
        for (ConfigurationItem entity : entities) {
            if (store.containsKey(entity.getId())) {
                throw new IllegalStateException("Entity " + entity.getId() + " already exists and cannot be created twice.");
            }
        }
    }

    private void storeEntities(final ConfigurationItem[] entities) {
        for (ConfigurationItem entity : entities) {
            store.put(entity.getId(), entity);
        }
    }

    public void clear() {
        store.clear();
    }
}
