package com.xebialabs.deployit.maven.cli;

import com.xebialabs.deployit.core.api.dto.FileUpload;
import com.xebialabs.deployit.core.api.dto.RepositoryObject;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.StringRequestEntity;
import org.apache.commons.httpclient.methods.multipart.FilePart;
import org.apache.commons.httpclient.methods.multipart.MultipartRequestEntity;
import org.apache.commons.httpclient.methods.multipart.Part;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.ws.rs.core.Response;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;


public class ImportHelper {

	private static final JAXBContext JAXB_CONTEXT;

	static {
		try {
			JAXB_CONTEXT = JAXBContext.newInstance(RepositoryObject.class);
		} catch (JAXBException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Check whether the packageToImport parameter refers to a file by checking for its existence.
	 * If it is not a file, it should be a package in the importablePackages directory on the server, and should be
	 * importable from there.
	 *
	 * @param httpClient
	 * @param packageToImport
	 * @return
	 * @throws IOException
	 */
	public RepositoryObject doImport(HttpClient httpClient, String baseUri, String packageToImport) {
		Response response;
		try {
			URL url = new URL(packageToImport);
			if (url.getProtocol().equals("file")) {
				return processFilePath(httpClient, baseUri, url.getFile());
			} else {
				PostMethod postMethod = new PostMethod(baseUri + "/importablePackages/import-url");
				postMethod.setRequestEntity(new StringRequestEntity(url.toString(), null, "UTF-8"));
				return executeAndExtract(httpClient, postMethod);
			}
		} catch (MalformedURLException e) {
			return processFilePath(httpClient, baseUri, packageToImport);
		} catch (UnsupportedEncodingException e) {
			throw new RuntimeException(e);
		}
	}

	private RepositoryObject executeAndExtract(HttpClient httpClient, PostMethod postMethod) {
		try {
			int i = httpClient.executeMethod(postMethod);
			if (i == 201) {
				// Ok, it was created.
				return extract(postMethod);
			} else throw new RuntimeException(postMethod.getResponseBodyAsString());
		} catch (IOException e) {
			throw new RuntimeException(e);
		} catch (JAXBException e) {
			throw new RuntimeException(e);
		}
	}

	protected RepositoryObject extract(PostMethod postMethod) throws JAXBException, IOException {
		return (RepositoryObject) JAXB_CONTEXT.createUnmarshaller().unmarshal(postMethod.getResponseBodyAsStream());
	}

	private RepositoryObject processFilePath(HttpClient httpClient, String baseUri, String path) {
		Response response;
		final File file = new File(path);
		if (file.exists()) {
			final FileUpload fileUpload = new FileUpload();
			try {
				fileUpload.setFileData(new FileInputStream(file));
				PostMethod postMethod = new PostMethod(baseUri + "/importablePackages/import-uploaded/" + file.getName());
				postMethod.setRequestEntity(new MultipartRequestEntity(new Part[]{
						new FilePart("fileData", file)
				}, postMethod.getParams()));
				return executeAndExtract(httpClient, postMethod);
			} catch (IOException e) {
				logger.error("The file \'{}\' exists, but could not be read for upload; perhaps it's not an archive.", file.toString());
				return null;
			}
		} else {
			if (!file.isAbsolute()) {
				PostMethod postMethod = new PostMethod(baseUri + "/importablePackages/import/" + path);
				return executeAndExtract(httpClient, postMethod);
			} else {
				logger.error("The file \'{}\' does not exist at the given absolute path", file.toString());
				return null;
			}
		}
	}

	private static final Logger logger = LoggerFactory.getLogger(ImportHelper.class);
}
