package com.xebialabs.deployit.maven.cli;

import com.google.common.base.Predicate;
import com.google.common.collect.Maps;
import com.xebialabs.deployit.core.api.dto.*;
import com.xebialabs.deployit.maven.Container;
import com.xebialabs.deployit.maven.Deployed;
import com.xebialabs.deployit.maven.cli.ssl.SelfSignedCertificateAcceptingSocketFactory;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.protocol.Protocol;
import org.apache.commons.httpclient.protocol.ProtocolSocketFactory;
import org.apache.commons.lang.StringUtils;
import org.apache.maven.plugin.logging.Log;

import javax.ws.rs.core.Response;
import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.text.SimpleDateFormat;
import java.util.*;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Iterables.any;
import static com.google.common.collect.Iterators.filter;
import static java.lang.String.format;

/**
 * Maven Client for Deployit.
 */
public class MavenCli {

	private Log logger;

	private MavenCliOptions options;
	private Authentication authentication;
	private MavenProxies proxies;

	private DeploymentClient deploymentClient;
	private DeployitClient deployitClient;
	private RepositoryClient repositoryClient;

	public MavenCli(MavenCliOptions options, Log log) throws Exception {
		this.logger = log;
		this.options = options;
		initialize();
	}

	private void initialize() throws Exception {
		authentication = createCredentials();
		setupSecureCommunications();
		authentication.init(this.options);
		attemptToConnectToServer(authentication);
		proxies = createAndRegisterProxies(authentication);

		deploymentClient = new DeploymentClient(proxies);
		deployitClient = new DeployitClient(proxies);
		repositoryClient = new RepositoryClient(proxies);
	}

	private Authentication createCredentials() {
		Authentication authentication = new Authentication();
		authentication.username = options.getUsername();
		authentication.password = options.getPassword();
		return authentication;
	}

	private void setupSecureCommunications() {
		if (options.isSecured()) {
			Protocol easyhttps = new Protocol("https", (ProtocolSocketFactory) new SelfSignedCertificateAcceptingSocketFactory(), 443);
			Protocol.registerProtocol("https", easyhttps);
		}
	}

	private MavenProxies createAndRegisterProxies(Authentication client) {
		return new MavenProxies(options, client);
	}


	private void attemptToConnectToServer(Authentication authentication) {
		String urlToConnectTo = options.getUrl();
		logger.info("Connecting to the Deployit server at " + urlToConnectTo + "...");
		try {
			final int responseCode = authentication.getHttpClient().executeMethod(new GetMethod(urlToConnectTo + "/server/info"));
			if (responseCode == 200) {
				logger.info("Succesfully connected.");
			} else if (responseCode == 401 || responseCode == 403) {
				throw new IllegalStateException("You were not authenticated correctly, did you use the correct credentials?");
			} else {
				throw new IllegalStateException("Could contact the server at " + urlToConnectTo + " but received an HTTP error code, " + responseCode);
			}
		} catch (MalformedURLException mue) {
			throw new IllegalStateException("Could not contact the server at " + urlToConnectTo, mue);
		} catch (IOException e) {
			throw new IllegalStateException("Could not contact the server at " + urlToConnectTo, e);
		}
	}

	public RepositoryObject create(Container ci) {
		final String id = ci.getId();
		try {
			return get(id);
		} catch (Exception e) {
			logger.debug(format("%s does not exist, create it", id));
			ConfigurationItemDto configurationItem = new ConfigurationItemDto(id, ci.getType());
			configurationItem.setValues(Maps.<String, Object>newHashMap(ci.getProperties()));
			final Response response = getProxies().getRepository().create(configurationItem.getId(), configurationItem);
			return checkForValidations(response);
		}
	}

	public void delete(String id) {
		try {
			logger.debug("Delete " + id);
			repositoryClient.delete(id);
		} catch (Exception e) {
			logger.debug(format("delete fails %s", id));
		}
	}

	public RepositoryObject get(String ciId) {
		final Response response = getProxies().getRepository().read(ciId);
		return checkForValidations(response);
	}

	private RepositoryObject checkForValidations(final Response response) {
		final ResponseExtractor responseExtractor = new ResponseExtractor(response);
		final RepositoryObject ci = responseExtractor.getEntity();
		if (!responseExtractor.isValidResponse() && !ci.getValidations().isEmpty()) {
			throw new IllegalStateException(format("Configuration item contained validation errors: {%s}", ci.getValidations()));
		}
		return ci;
	}

	public RepositoryObject importPackage(File darFile) {
		return deployitClient.importPackage(darFile.getPath());
	}

	public String deploy(String source, String target, List<Deployed> configuredDeployeds) {
		Deployment deployment;
		String previouslyDeployedApplicationId = null;
		boolean initialDeployment = isInitialDeployment(source, target);
		if (initialDeployment) {
			logger.info("initial Deployment");
			deployment = deploymentClient.prepareInitial(source, target);
			if (!options.isExplicitMappings()) {
				logger.debug("generateAllDeployeds");
				deployment = deploymentClient.generateAllDeployeds(deployment);
			}
		} else {
			logger.info("upgrade Deployment");
			String deployedApplicationId = getDeployedApplicationId(source, target);
			previouslyDeployedApplicationId = getPreviousDeployedPackage(deployedApplicationId);
			deployment = deploymentClient.prepareUpgrade(source, deployedApplicationId);
		}


		if (options.isExplicitMappings()) {
			logger.debug("use explicits deployeds");
			for (Deployed configuredDeployed : configuredDeployeds) {
				checkNotNull(configuredDeployed.getId(), "id is mandatory in the explicit deployed mode");
				String type = configuredDeployed.getType();
				checkNotNull(type, "type is mandatory in the explicit deployed mode");

				final String deployableId = configuredDeployed.getDeployable(source);
				final String containerId = configuredDeployed.getContainer();
				final boolean exist = any(deployment.getDeployeds(), new Predicate<ConfigurationItemDto>() {
					@Override
					public boolean apply(ConfigurationItemDto input) {
						return input.getValues().get("deployable").equals(deployableId)
								&& input.getValues().get("container").equals(containerId);
					}
				});
				if (exist) {
					logger.debug(format(" deployed  %s %s %s already exists", deployableId, containerId, type));
				} else {
					logger.debug(format(" generateSingleDeployed %s %s %s", deployableId, containerId, type));
					deployment = deploymentClient.generateSingleDeployed(deployableId, containerId, type, deployment);
				}

			}
		}

		if (configuredDeployeds != null) {
			logger.debug("update the generated deployeds with the configured deployeds");
			final List<ConfigurationItemDto> deployeds = deployment.getDeployeds();
			for (ConfigurationItemDto configurationItem : deployeds) {
				logger.debug(" check " + configurationItem);
				final String id = configurationItem.getId();
				final Iterator<Deployed> deployedIterator = filter(configuredDeployeds.iterator(), new Predicate<Deployed>() {
					@Override
					public boolean apply(Deployed input) {
						return input.getId().equals(id);
					}
				});
				if (deployedIterator.hasNext()) {
					updateConfigurationItemValues(configurationItem, deployedIterator.next());
				} else {
					logger.debug(" no configured deployed found with id " + configurationItem.getId());
				}
			}
		}

		if (logger.isDebugEnabled()) {
			logger.debug(" dump Deployeds");
			for (ConfigurationItemDto itemDto : deployment.getDeployeds()) {
				logger.debug(" - " + itemDto);
			}
		}

		try {
			logger.debug("validate");
			deployment = deploymentClient.validate(deployment);
		} catch (RuntimeException e) {
			logger.error(" RuntimeException: " + e.getMessage());
			if (options.isFailIfNoStepsAreGenerated() || !e.getMessage().contains("The task did not deliver any steps")) {
				throw e;
			}
			return null;
		}

		int validationMessagesFound = 0;
		for (ConfigurationItemDto configurationItem : deployment.getDeployeds()) {
			for (Message msg : configurationItem.getValidations()) {
				logger.error(format("Validation error found on '%s' on field '%s': %s, %s", configurationItem.getId(), msg.getField(), msg.getMessage(), configurationItem));
				logger.error(format(" %s", configurationItem));
				validationMessagesFound++;
			}
		}

		if (validationMessagesFound > 0) {
			throw new IllegalStateException(format("Validation errors (%d) have been found", validationMessagesFound));
		}

		logger.debug("deploy");
		executeTask(deploymentClient.deploy(deployment).getTaskId());
		return previouslyDeployedApplicationId;
	}


	private void updateConfigurationItemValues(ConfigurationItemDto configurationItem, Deployed configuredDeployed) {
		logger.debug(format(" update values of %s with %s", configuredDeployed.getId(), configuredDeployed.getValues()));
		configurationItem.getValues().putAll(configuredDeployed.getValues());
		if (!configuredDeployed.getPlaceholders().isEmpty()) {
			if (!configurationItem.getValues().containsKey("placeholders")) {
				configurationItem.getValues().put("placeholders", new HashMap<String, String>());
			}
			Map<String, String> placeholders = (Map<String, String>) configurationItem.getValues().get("placeholders");
			placeholders.putAll(configuredDeployed.getPlaceholders());
		}
		logger.debug(configurationItem.getValues().toString());
	}

	public void undeployAndWait(String source) {
		logger.info("   undeployAndWait " + source);
		String taskId = this.deploymentClient.undeploy(source).getTaskId();
		executeTask(taskId);
	}


	private boolean executeTask(String taskId) {
		if (options.isSkipMode()) {
			logger.info("skip mode, skip all the steps");
			getDeployitClient().skipSteps(taskId, range(1, getDeployitClient().retrieveTaskInfo(taskId).getNrOfSteps() + 1));
		}
		checkTaskState(taskId);
		if (options.isTestMode()) {
			logger.info("test mode, cancel task " + taskId);
			getDeployitClient().cancelTask(taskId);
			return false;
		}

		try {
			logger.info("Start deployment task " + taskId);
			getDeployitClient().startTaskAndWait(taskId);
			checkTaskState(taskId);
			return true;
		} catch (RuntimeException e) {
			if (taskId != null) {
				logger.info(format("on Error (%s), cancel task %s", e.getMessage(), taskId));
				getDeployitClient().cancelTask(taskId);
			}
			throw e;
		}

	}

	private boolean isInitialDeployment(String source, String target) {
		String deployedApplicationId = getDeployedApplicationId(source, target);
		logger.debug("  deployedApplicationId " + deployedApplicationId);
		try {
			get(deployedApplicationId);
			return false;
		} catch (Exception e) {
			return true;
		}
	}


	private String getDeployedApplicationId(String source, String target) {
		//target = "Environments/DefaultEnvironment/deployit-petclinic";
		return target + "/" + StringUtils.split(source, "/")[1];
	}

	private String getPreviousDeployedPackage(String target) {
		final RepositoryObject dp = repositoryClient.read(target);
		final Map<String, Object> values = dp.getValues();
		Object source = values.get("version");
		return (source == null ? null : source.toString());
	}

	private void checkTaskState(String taskId) {
		final TaskInfo taskInfo = getDeployitClient().retrieveTaskInfo(taskId);
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd hh:mm:ss");
		logger.info(format("%s Label      %s", taskId, taskInfo.getLabel()));
		logger.info(format("%s State      %s %d/%d", taskId, taskInfo.getState(), taskInfo.getCurrentStepNr(), taskInfo.getNrOfSteps()));
		if (taskInfo.getStartDate() != null) {
			final GregorianCalendar startDate = (GregorianCalendar) taskInfo.getStartDate();
			logger.info(format("%s Start      %s", taskId, sdf.format(startDate.getTime())));
		}

		if (taskInfo.getCompletionDate() != null) {
			final GregorianCalendar completionDate = (GregorianCalendar) taskInfo.getCompletionDate();
			logger.info(format("%s Completion %s", taskId, sdf.format(completionDate.getTime())));
		}

		StringBuilder sb = new StringBuilder();
		for (int i = 1; i <= taskInfo.getNrOfSteps(); i++) {
			final Response stepInfoResponse = getProxies().getTaskRegistry().getStepInfo(taskId, i, null);
			final StepInfo stepInfo = new ResponseExtractor(stepInfoResponse).getEntity();
			final String description = stepInfo.getDescription();
			final String log = stepInfo.getLog();
			String stepInfoMessage;
			if (StringUtils.isEmpty(log) || description.equals(log)) {
				stepInfoMessage = format("%s step #%d %s\t%s", taskId, i, stepInfo.getState(), description);
			} else {
				stepInfoMessage = format("%s step #%d %s\t%s\n%s", taskId, i, stepInfo.getState(), description, log);
			}

			logger.info(stepInfoMessage);
			if ("FAILED".endsWith(stepInfo.getState()))
				sb.append(stepInfoMessage);
		}

		if ("STOPPED".equals(taskInfo.getState()))
			throw new IllegalStateException(format("Errors when executing task %s: %s", taskId, sb));
	}


	private Integer[] range(int begin, int end) {
		Integer[] result = new Integer[end - begin];
		for (int i = begin; i < end; i++) {
			result[i - begin] = i;
		}
		return result;
	}


	private DeployitClient getDeployitClient() {
		return deployitClient;
	}

	private DeploymentClient getDeploymentClient() {
		return deploymentClient;
	}

	private MavenProxies getProxies() {
		return proxies;
	}


}
