package com.xebialabs.deployit.maven.packager;

import com.xebialabs.deployit.maven.Deployable;
import org.apache.commons.io.IOUtils;
import org.apache.maven.archiver.MavenArchiveConfiguration;
import org.apache.maven.archiver.MavenArchiver;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.archiver.jar.JarArchiver;
import org.codehaus.plexus.util.DirectoryScanner;
import org.codehaus.plexus.util.FileUtils;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.List;
import java.util.Map;
import java.util.jar.Attributes;
import java.util.jar.Manifest;

import static com.google.common.base.Strings.isNullOrEmpty;
import static java.lang.String.format;

public class ManifestPackager {
	public static final String DEPLOYMENT_PACKAGE_DIR = "deployment-package";
	public static final String APPLICATION = "CI-Application";
	public static final String VERSION = "CI-Version";

	private final MavenProject project;
	private final Log log;
	private final boolean timestampedVersion;

	private final File outputDirectory;
	private final File targetDirectory;

	private final Manifest manifest = new Manifest();

	public ManifestPackager(MavenProject project, Log log, boolean timestampedVersion) {
		this.project = project;
		this.log = log;
		this.timestampedVersion = timestampedVersion;
		this.outputDirectory = new File(project.getBuild().getDirectory());
		this.targetDirectory = new File(outputDirectory, DEPLOYMENT_PACKAGE_DIR + File.separator + project.getArtifactId() + File.separator + project.getVersion());
		this.targetDirectory.mkdirs();

	}

	public void addDeployables(List<Deployable> deployables) {
		for (Deployable deployable : deployables)
			addDeployable(deployable);
	}

	private void addDeployable(Deployable deployable) {
		log.debug("addDeployable " + deployable);
		deployable.consolidate(project);
		final Map<String, Attributes> entries = manifest.getEntries();
		final Attributes attributes = new Attributes();

		attributes.putValue("CI-Type", deployable.getType());
		attributes.putValue("CI-Name", deployable.getName());
		for (Map.Entry<String, String> me : deployable.getValues().entrySet()) {
			attributes.putValue(me.getKey(), me.getValue());
		}
		entries.put(deployable.getEntryName(), attributes);

		if (isNullOrEmpty(deployable.getFileSystemLocation())) {
			return;
		}

		final File location = new File(deployable.getFileSystemLocation());
		try {
			if (!location.exists())
				throw new IOException(format("source location %s doesn't exist", location));

			if (location.isFile()) {
				final File deployableTargetFile = new File(targetDirectory, deployable.getEntryName());
				deployableTargetFile.getParentFile().mkdirs();
				log.debug(format(" copy file %s to %s", location, deployableTargetFile));
				FileUtils.copyFile(location, deployableTargetFile);
			} else {
				final File deployableTargetFile = targetDirectory;
				log.debug(format(" copy dir  %s to %s", location, deployableTargetFile));
				copyToDir(location, deployableTargetFile);
			}
		} catch (IOException e) {
			throw new RuntimeException(format("Fail to copy of %s to %s ", location, targetDirectory), e);

		}
	}

	public void perform() {
		final Attributes mainAttributes = manifest.getMainAttributes();
		mainAttributes.putValue("Manifest-Version", "1.0");
		mainAttributes.putValue("Deployit-Package-Format-Version", "1.3");
		mainAttributes.putValue(APPLICATION, project.getArtifactId());
		final String pomVersion = project.getVersion();

		String darVersion = pomVersion;
		if (timestampedVersion) {
			SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMdd-HHmmss");
			darVersion = pomVersion + "-" + dateFormat.format(System.currentTimeMillis());
			//SNAPSHOT is removed from the timestamped version (as maven deploy does).
			darVersion = darVersion.replace("-SNAPSHOT", "");
		}

		mainAttributes.putValue(VERSION, darVersion);

		final File meta_inf = new File(targetDirectory, "META-INF");
		meta_inf.mkdirs();
		File manifestFile = new File(meta_inf, "MANIFEST.MF");
		log.info("Generate manifest file " + manifestFile.getAbsolutePath());
		FileOutputStream fos = null;
		try {
			dumpManifest();
			fos = new FileOutputStream(manifestFile);
			manifest.write(fos);
		} catch (IOException e) {
			new RuntimeException("generation of the manifest file failed", e);
		} finally {
			IOUtils.closeQuietly(fos);
		}
	}

	private void dumpManifest() throws IOException {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		try {
			manifest.write(baos);
		} finally {
			IOUtils.closeQuietly(baos);
		}
		log.debug(new String(baos.toByteArray()));
	}

	public void seal() throws MojoExecutionException {
		try {
			File darFile = getDarFile();
			log.info("Seal the archive in " + darFile);

			final MavenArchiver mvnArchiver = new MavenArchiver();
			mvnArchiver.setArchiver(new JarArchiver());
			mvnArchiver.setOutputFile(darFile);

			mvnArchiver.getArchiver().addDirectory(targetDirectory);

			final File manifestFile = getManifestFile();
			log.debug("set Manifest file of the archive " + manifestFile);
			mvnArchiver.getArchiver().setManifest(manifestFile);

			mvnArchiver.createArchive(project, new MavenArchiveConfiguration());

			project.getArtifact().setFile(darFile);

		} catch (Exception e) {
			throw new MojoExecutionException("Error when sealing DAR", e);
		}
	}

	public File getDarFile() {
		return new File(project.getBuild().getDirectory(), project.getBuild().getFinalName() + ".dar");
	}

	public File getManifestFile() {
		return new File(targetDirectory, "META-INF/MANIFEST.MF");
	}

	private void copyToDir(File sourceDirectory, File targetDir) throws IOException {
		File destination = new File(targetDir, sourceDirectory.getName());

		DirectoryScanner scanner = new DirectoryScanner();
		scanner.addDefaultExcludes();
		scanner.setBasedir(sourceDirectory.getPath());
		scanner.scan();

		String[] files = scanner.getIncludedFiles();
		if (files != null && files.length > 0) {
			for (int i = 0; i < files.length; i++) {
				File sourceFile = new File(scanner.getBasedir(), files[i]);
				File targetFile = new File(destination, files[i]);
				log.debug("  copy " + files[i] + " to " + targetFile);
				FileUtils.copyFile(sourceFile, targetFile);
			}
		}
	}
}
