package com.xebialabs.deployit.maven;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import com.xebialabs.deployit.client.Deployable;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.project.MavenProject;

import java.io.File;
import java.lang.reflect.Field;
import java.util.Map;
import java.util.Set;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Strings.emptyToNull;
import static com.google.common.base.Strings.isNullOrEmpty;
import static com.google.common.collect.Maps.uniqueIndex;
import static java.lang.String.format;

public class MavenDeployable extends Deployable {

	private String groupId;

	private String artifactId;

	private String classifier;

	private MavenProject project;

	public void addValue(String name, String value) {
		checkNotNull(name);
		checkNotNull(value, format("null value for %s", name));
		try {
			if ("location".equals(name)) {
				setLocation(new File(value));
			} else {
				Field declaredField = getField(name);
				declaredField.set(this, value);
			}
		} catch (NoSuchFieldException e) {
			values.put("CI-" + name, value);
		} catch (IllegalAccessException e) {
			throw new RuntimeException(format("Cannot set %s value on field %s", name, value), e);
		}
	}

	public String getId() {
		return Joiner.on(":").skipNulls().join(groupId, emptyToNull(classifier), artifactId);
	}

	private Field getField(String name) throws NoSuchFieldException {
		try {
			return MavenDeployable.class.getDeclaredField(name);
		} catch (NoSuchFieldException nsfe) {
			return Deployable.class.getDeclaredField(name);
		}
	}

	@Override
	public String getName() {
		return Strings.isNullOrEmpty(name) ? artifactId : name;
	}

	@Override
	public File getLocation() {
		if (location != null) {
			if (location.isAbsolute())
				return location;
			else
				return new File(project.getBasedir(), location.getPath());
		} else if (isMavenArtefact()) {
			final String id = getId();
			Artifact artifact = (Artifact) project.getArtifactMap().get(id);
			final Set<Artifact> dependencyArtifacts = project.getDependencyArtifacts();
			if (dependencyArtifacts != null) {
				final Map<String, Artifact> idToArtifact = uniqueIndex(dependencyArtifacts, new Function<Artifact, String>() {
					@Override
					public String apply(Artifact input) {
						return Joiner.on(":").skipNulls().join(input.getGroupId(), emptyToNull(input.getClassifier()), input.getArtifactId());
					}
				});
				if (idToArtifact.containsKey(getId())) {
					artifact = idToArtifact.get(getId());
				}
			}

			if (artifact == null)
				throw new IllegalStateException(format("The artifact %s not found in the project dependencies %s", getId(), dependencyArtifacts));

			final File file = artifact.getFile();
			if (file == null) {
				throw new IllegalStateException(format("Associated file of %s is empty", artifact));
			}

			return file;
		}
		return null;
	}

	private boolean isMavenArtefact() {
		return !isNullOrEmpty(groupId) && !isNullOrEmpty(artifactId);
	}

	public void setGroupId(String groupId) {
		this.groupId = groupId;
	}

	public void setArtifactId(String artifactId) {
		this.artifactId = artifactId;
	}

	public void setProject(MavenProject project) {
		this.project = project;
	}

	@Override
	public String toString() {
		return "MavenDeployable{" +
				"artifactId='" + artifactId + '\'' +
				", classifier='" + classifier + '\'' +
				", groupId='" + groupId + '\'' +
				"} " + super.toString();
	}

	public String getArtifactId() {
		return artifactId;
	}
}
