package com.xebialabs.deployit.maven.packager;

import java.io.*;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import org.apache.commons.io.FileUtils;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;
import org.jdom2.*;
import org.jdom2.input.StAXStreamBuilder;
import org.jdom2.output.Format;
import org.jdom2.output.XMLOutputter;
import com.google.common.base.Strings;
import com.google.common.io.Closeables;

import com.xebialabs.deployit.maven.MavenDeployable;
import com.xebialabs.deployit.maven.XmlFragment;
import com.xebialabs.deployit.maven.converter.Files;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.maven.MavenDeployable.CiRefProperty;
import static java.lang.String.format;

public class MavenDarPackager {

    private final Log log;

    private final List<MavenDeployable> deployables = newArrayList();
    private final List<XmlFragment> properties = newArrayList();
    private MavenProject project;
    private final String applicationName;
    private final boolean timestampedVersion;


    public MavenDarPackager(MavenProject project, String applicationName, boolean timestampedVersion, Log log) {
        this.project = project;
        this.applicationName = Strings.isNullOrEmpty(applicationName) ? project.getArtifactId() : applicationName;
        this.timestampedVersion = timestampedVersion;
        this.log = log;
    }

    public void addProperties(List<XmlFragment> properties) {
        if (properties !=null && !properties.isEmpty()) {
            this.properties.addAll(properties);
        }
    }

    public void addDeployables(Collection<? extends MavenDeployable> deployables) {
        this.deployables.addAll(deployables);
    }

    public File perform() {
        File targetDir = new File(project.getBuild().getDirectory());
        File darFile = new File(targetDir, format("%s-%s.dar", project.getArtifactId(), project.getVersion()));
        if (darFile.exists()) {
            return darFile;
        }

        File workingDirectory = createWorkingDirectory(targetDir);
        log.debug(" prepare exploded dar in working directory " + workingDirectory);
        generateExplodedDar(workingDirectory);

        log.debug(" create the jar " + darFile);
        Files.jar(darFile, workingDirectory);

        return darFile;
    }

    private void generateExplodedDar(File workingDirectory) {
        Element manifestElm = new Element("udm.DeploymentPackage");
        manifestElm.setAttribute("version", getCIVersion(project, timestampedVersion));
        manifestElm.setAttribute("application", applicationName);

        List<Content> propertyElms = marshallProperties();
        if (!propertyElms.isEmpty()) {
             manifestElm.addContent(propertyElms);
        }

        Element deployablesElm = new Element("deployables");

        for (MavenDeployable deployable : deployables) {
            deployablesElm.addContent(marshallDeployable(deployable, workingDirectory));
        }

        manifestElm.addContent(deployablesElm);
        writeManifest(manifestElm, workingDirectory);
    }

    private List<Content> marshallProperties() {
        XMLInputFactory factory = XMLInputFactory.newInstance();
        List<Content> contents = newArrayList();
        StAXStreamBuilder builder = new StAXStreamBuilder();
        for (XmlFragment property : properties) {
            try {
                XMLStreamReader xmlStreamReader = factory.createXMLStreamReader(new StringReader(property.getFragment()));
                xmlStreamReader.next();
                Content fragment = builder.fragment(xmlStreamReader);
                contents.add(fragment);
            } catch (JDOMException e) {
                throw new RuntimeException(e);
            } catch (XMLStreamException e) {
                throw new RuntimeException(e);
            }
        }

        return contents;
    }

    private Content marshallDeployable(final MavenDeployable deployable, File workingDirectory) {
        Element deployableElm = new Element(deployable.getType());
        deployableElm.setAttribute("name", deployable.getName());
        if (deployable.getLocation() != null) {
            String relFilePath = stageArtifact(deployable, workingDirectory);
            deployableElm.setAttribute("file", relFilePath);
        }

        Map<String, Object> values = deployable.getValues();
        for (String name : values.keySet()) {
            Element propElm = new Element(name);
            List<Content> contentElms = newArrayList();
            Object value = values.get(name);
            if (value instanceof List) {
                contentElms = marshallListValues((List<?>) value, workingDirectory);
            } else if (value instanceof CiRefProperty) {
                propElm.setAttribute("ref", ((CiRefProperty) value).getRef());
            } else if (value instanceof Map) {
                contentElms = marshallMapEntries((Map<String, String>) value);
            } else if (value instanceof MavenDeployable) {
                contentElms.add(marshallDeployable((MavenDeployable)value, workingDirectory));
            } else {
                contentElms.add(new Text((String) value));
            }

            propElm.addContent(contentElms);
            deployableElm.addContent(propElm);
        }

        if (!deployable.getTags().isEmpty()) {
            Element propElm = new Element("tags");
            propElm.addContent(marshallListValues(deployable.getTags(), workingDirectory));
            deployableElm.addContent(propElm);
        }

        return deployableElm;
    }

    private String stageArtifact(final MavenDeployable deployable, File workingDirectory) {
        final File location = deployable.getLocation();
        File artifactWorkingDirectory = new File(workingDirectory, deployable.getName());
        artifactWorkingDirectory.mkdir();
        if (location.isFile()) {
            log.debug(format(" copy file %s to %s", location, artifactWorkingDirectory));
            Files.copyFileTo(location, artifactWorkingDirectory);

        } else {
            log.debug(format(" copy dir  %s to %s", location, artifactWorkingDirectory));
            Files.copyDirectoryTo(location, artifactWorkingDirectory);
        }

        return deployable.getName() + "/" + location.getName();
    }

    private List<Content> marshallMapEntries(final Map<String, String> map) {
        List<Content> elms = newArrayList();
        for (Map.Entry<String, String> entry : map.entrySet()) {
            Element elm = new Element("entry");
            elm.setAttribute("key", entry.getKey());
            elm.addContent(new Text(entry.getValue()));
            elms.add(elm);
        }
        return elms;
    }

    private List<Content> marshallListValues(final List<?> list, File workingDirectory) {
        List<Content> elms = newArrayList();

        for (Object o : list) {
            if (o instanceof CiRefProperty) {
                Element ciRefElm = new Element("ci");
                ciRefElm.setAttribute("ref", ((CiRefProperty) o).getRef());
                elms.add(ciRefElm);
            } else if (o instanceof MavenDeployable) {
                elms.add(marshallDeployable((MavenDeployable) o, workingDirectory));
            } else {
                Element valueElm = new Element("value");
                valueElm.addContent(new Text((String) o));
                elms.add(valueElm);
            }
        }

        return elms;
    }

    private File createWorkingDirectory(final File targetDir) {
        File workingDirectory = new File(targetDir, "deployit-working-dir");
        if (workingDirectory.exists()) {
            try {
                FileUtils.deleteDirectory(workingDirectory);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
        workingDirectory.mkdirs();
        return workingDirectory;
    }

    private void writeManifest(Element manifestElm, File workingDirectory) {
        FileOutputStream outputStream = null;
        try {
            File manifest = new File(workingDirectory, "deployit-manifest.xml");
            log.info("Generate manifest file " + manifest.getAbsolutePath());
            outputStream = new FileOutputStream(manifest);

            XMLOutputter xmlOutput = new XMLOutputter(Format.getPrettyFormat());
            Document doc = new Document(manifestElm);
            log.debug(xmlOutput.outputString(doc));
            xmlOutput.output(doc, outputStream);

        } catch (FileNotFoundException e) {
            throw new RuntimeException(e);
        } catch (IOException e) {
            throw new RuntimeException("generation of the manifest file failed", e);
        } finally {
            Closeables.closeQuietly(outputStream);
        }
    }

    private static String getCIVersion(MavenProject project, boolean timestampedVersion) {
        String ciVersion = project.getVersion();
        if (timestampedVersion) {
            SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMdd-HHmmss");
            ciVersion = project.getVersion() + "-" + dateFormat.format(System.currentTimeMillis());
            //SNAPSHOT is removed from the timestamped version (as maven deploy does).
            ciVersion = ciVersion.replace("-SNAPSHOT", "");
        }
        return ciVersion;
    }

}
