package com.xebialabs.deployit.deployment.orchestrator;

import static com.google.common.collect.Lists.transform;
import static com.xebialabs.deployit.plugin.api.deployment.execution.Plans.interleaved;

import java.util.Collection;
import java.util.List;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.google.common.collect.ImmutableListMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Multimaps;
import com.xebialabs.deployit.plugin.api.deployment.execution.Plan;
import com.xebialabs.deployit.plugin.api.deployment.execution.Plans;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.udm.CompositePackage;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.Version;
import com.xebialabs.deployit.server.api.orchestrator.Orchestrator;

@Orchestrator.Metadata(name = "composite-package")
public class CompositePackageOrchestrator implements Orchestrator {
	@Override
	public Plan orchestrate(DeltaSpecification specification) {
		Version version = specification.getDeployedApplication().getVersion();
		if (version instanceof CompositePackage) {
			CompositePackage p = (CompositePackage) version;
			
			final List<String> ids = transform(p.getPackages(), new Function<Version, String>() {
				@Override
				public String apply(Version input) {
					return input.getId();
				}
			});

			final ImmutableListMultimap<String,Delta> index = Multimaps.index(specification.getDeltas(), new Function<Delta, String>() {
				public String apply(Delta input) {
					final String id = getDeployed(input).getDeployable().getId();
					Collection<String> filter = Collections2.filter(ids, new Predicate<String>() {
						@Override
						public boolean apply(String input) {
							return id.startsWith(input);
						}
					});
					if (filter.isEmpty() || filter.size() > 1) {
						throw new IllegalStateException(String.format("Found no or more than 1 DeploymentPackage (%s) which could have been the source of [%s]", filter, id));
					}
					return Iterables.get(filter, 0);
				}
			});
			
			return Plans.serial(transform(ids, new Function<String, Plan>() {
				@Override
				public Plan apply(String input) {
					return interleaved(index.get(input));
				}
			}));
			
		} else {
			return new DefaultOrchestrator().orchestrate(specification);
		}
	}

	private Deployed<?, ?> getDeployed(Delta input) {
		return input.getDeployed() != null ? input.getDeployed() : input.getPrevious();
	}

}
