/*
 * Copyright (c) 2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of Xebialabs Software and Documentation is subject to the Personal
 * License Agreement.
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * ‚ÄúDocumentation‚Äù means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the Xebialabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to  (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.hostsession;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

import com.xebialabs.deployit.exception.RuntimeIOException;

/**
 * Contains a number of convenience methods for working with HostFile objects.
 */
public class HostFileUtils {

	/**
	 * Copies a file or directory.
	 * 
	 * @param src
	 *            the source file or directory.
	 * @param dst
	 *            the destination file or directory. If it exists it must be of the same type as the source. Its parent
	 *            directory must exist.
	 * @throws RuntimeIOException
	 *             if an I/O error occurred
	 */
	public static void copy(HostFile src, HostFile dst) {
		if (src.isDirectory()) {
			copyDirectory(src, dst);
		} else {
			copyFile(src, dst);
		}

	}

	/**
	 * Copies a regular file.
	 * 
	 * @param srcFile
	 *            the source file. Must exists and must not be a directory.
	 * @param dstFile
	 *            the destination file. May exists but must not be a directory. Its parent directory must exist.
	 * @throws RuntimeIOException
	 *             if an I/O error occurred
	 */
	public static void copyFile(HostFile srcFile, HostFile dstFile) throws RuntimeIOException {
		// check source file
		if (!srcFile.exists()) {
			throw new RuntimeIOException("Source file " + srcFile + " does not exist");
		}
		if (srcFile.isDirectory()) {
			throw new RuntimeIOException("Source file " + srcFile + " exists but is a directory");
		}

		// check destination file
		if (dstFile.exists()) {
			if (dstFile.isDirectory()) {
				throw new RuntimeIOException("Destination file " + dstFile + " exists but is a directory");
			}
			if (logger.isDebugEnabled())
				logger.debug("About to overwrite existing file " + dstFile);
		}

		if (logger.isDebugEnabled())
			logger.debug("Copying contents of regular file " + srcFile + " to " + dstFile);
		InputStream in = srcFile.get();
		try {
			dstFile.put(in, srcFile.length());
		} finally {
			IOUtils.closeQuietly(in);
		}
	}

	/**
	 * Copies a directory recursively.
	 * 
	 * @param srcDir
	 *            the source directory. Must exist and must not be a directory.
	 * @param dstDir
	 *            the destination directory. May exists but must a directory. Its parent directory must exist.
	 * @throws RuntimeIOException
	 *             if an I/O error occurred
	 */
	public static void copyDirectory(HostFile srcDir, HostFile dstDir) throws RuntimeIOException {
		// check source directory
		if (!srcDir.exists()) {
			throw new RuntimeIOException("Source directory " + srcDir + " does not exist");
		}
		if (!srcDir.isDirectory()) {
			throw new RuntimeIOException("Source directory " + srcDir + " exists but is not a directory");
		}

		// check destination directory, and create if it does not exist yet
		if (dstDir.exists()) {
			if (!dstDir.isDirectory()) {
				throw new RuntimeIOException("Destination directory " + dstDir + " already exists and is not a directory");
			}
			if (logger.isDebugEnabled())
				logger.debug("About to copy files into existing directory " + dstDir);
		} else {
			if (logger.isDebugEnabled())
				logger.debug("Creating destination directory " + dstDir);
			dstDir.mkdir();
		}

		if (logger.isDebugEnabled())
			logger.debug("Copying contents of directory " + srcDir + " to " + dstDir);
		List<HostFile> srcFiles = srcDir.listFiles();
		for (HostFile srcFile : srcFiles) {
			HostFile dstFile = dstDir.getFile(srcFile.getName());
			copy(srcFile, dstFile);
		}
	}

	/**
	 * Copies the contents of a string to a {@link HostFile}.
	 * 
	 * @param sourceString
	 *            the string to copy.
	 * @param targetFile
	 *            the host file to copy to.
	 */
	public static void putStringToHostFile(String sourceString, HostFile targetFile) {
		byte[] bytes;
		try {
			bytes = sourceString.getBytes("UTF-8");
		} catch (UnsupportedEncodingException exc) {
			throw new RuntimeIOException(exc);
		}

		InputStream in = new ByteArrayInputStream(bytes);
		try {
			targetFile.put(in, bytes.length);
		} finally {
			IOUtils.closeQuietly(in);
		}
	}

	/**
	 * Reads the contents of a {@link HostFile} as a string.
	 * 
	 * @param sourceHostFile
	 *            the host file to read.
	 * @return the contents of the host file.
	 */
	public static String getHostFileAsString(HostFile sourceHostFile) {
		String content;
		OutputStream os = new ByteArrayOutputStream();
		try {
			sourceHostFile.get(os);
			String unformattedContent = os.toString();
			content = new String(unformattedContent.getBytes("UTF-8"));
		} catch (UnsupportedEncodingException exc) {
			throw new RuntimeIOException(exc);
		} finally {
			IOUtils.closeQuietly(os);
		}
		return content;
	}

	/**
	 * Unzips a host file.
	 * 
	 * @param zip
	 *            the file to unzip.
	 * @param dir
	 *            the directory to unzip to.
	 */
	public static void unzip(HostFile zip, HostFile dir) {
		if (!dir.isDirectory()) {
			throw new IllegalArgumentException(dir + " is not a directory");
		}
		unzip(zip.getPath(), zip.get(), dir);
	}

	private static void unzip(String zipPath, InputStream zipStream, HostFile dstDir) {
		if (logger.isDebugEnabled())
			logger.debug("Unzipping " + zipPath + " to " + dstDir);

		ZipInputStream zipIn = new ZipInputStream(zipStream);
		try {
			try {
				ZipEntry entry;
				while ((entry = zipIn.getNextEntry()) != null) {
					uploadZipEntry(zipIn, entry, dstDir);
				}
			} catch (IOException exc) {
				throw new RuntimeIOException("Cannot unzip " + zipPath + " to " + dstDir, exc);
			}
		} finally {
			IOUtils.closeQuietly(zipIn);
		}
	}

	private static void uploadZipEntry(ZipInputStream zipIn, ZipEntry entry, HostFile dir) {
		String[] pathElements = StringUtils.split(entry.getName(), "/");
		String dstFileSeparator = dir.getSession().getHostOperatingSystem().getFileSeparator();
		String dstPath = StringUtils.join(pathElements, dstFileSeparator);
		HostFile dstFile = dir.getFile(dstPath);

		if (logger.isDebugEnabled())
			logger.debug("Unzipping " + entry.getName() + " to " + dstFile.getPath());

		if (entry.isDirectory()) {
			dstFile.mkdirs();
		} else {
			dstFile.getParentFile().mkdirs();
			dstFile.put(zipIn, entry.getSize());
		}
	}

	private static Logger logger = Logger.getLogger(HostFileUtils.class);

}
