/*
 * Copyright (c) 2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of Xebialabs Software and Documentation is subject to the Personal
 * License Agreement.
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * ‚ÄúDocumentation‚Äù means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the Xebialabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to  (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.hostsession.common;

import java.io.File;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Random;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.core.io.Resource;

import com.xebialabs.deployit.ci.OperatingSystemFamily;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.hostsession.HostFile;
import com.xebialabs.deployit.hostsession.HostSession;

/**
 * Abstract base class with common methods used by actual implementations of {@link HostSession}.
 */
public abstract class AbstractHostSession implements HostSession {

	private OperatingSystemFamily os;

	private String temporaryDirectoryPath;

	private HostFile sessionTemporaryDirectory;

	public static final long MAX_TEMP_RETRIES = 100;

	public AbstractHostSession(OperatingSystemFamily os, String temporaryDirectoryPath) {
		this.os = os;
		this.temporaryDirectoryPath = temporaryDirectoryPath;
	}

	public OperatingSystemFamily getHostOperatingSystem() {
		return os;
	}

	public void close() {
		String doNotCleanUpTemporaryFiles = System.getProperty("com.xebia.ad.donotcleanuptemporaryfiles");
		boolean doNotCleanUp = (doNotCleanUpTemporaryFiles != null && doNotCleanUpTemporaryFiles.equalsIgnoreCase("true"));
		if (!doNotCleanUp) {
			cleanupTemporaryFiles();
		}
	}

	protected synchronized HostFile getTemporaryDirectory() throws RuntimeIOException {
		if (sessionTemporaryDirectory == null) {
			for (;;) {
				HostFile temporaryDirectory = getFile(temporaryDirectoryPath);
				Random r = new Random();
				DateFormat dateFormat = new SimpleDateFormat("yyyyMMdd'T'HHmmssSSS");
				String prefix = "deployit-" + dateFormat.format(new Date());
				String infix = "";
				String suffix = ".tmp";
				for (int i = 0; i < MAX_TEMP_RETRIES; i++) {
					HostFile f = getFile(temporaryDirectory, prefix + infix + suffix);
					if (!f.exists()) {
						f.mkdir();
						sessionTemporaryDirectory = f;
						logger.info("Created session temporary directory " + sessionTemporaryDirectory);
						return sessionTemporaryDirectory;
					}
					infix = "-" + Long.toString(Math.abs(r.nextLong()));
				}
				throw new RuntimeIOException("Cannot create session temporary directory on " + this);
			}
		}
		return sessionTemporaryDirectory;
	}

	public void cleanupTemporaryFiles() {
		if (sessionTemporaryDirectory != null) {
			sessionTemporaryDirectory.deleteRecursively();
			logger.info("Removed session temporary directory " + sessionTemporaryDirectory);
		}
	}

	public HostFile getTempFile(String nameTemplate) throws RuntimeIOException {
		String prefix, suffix;

		if (nameTemplate != null) {
			int pos = nameTemplate.lastIndexOf('/');
			if (pos != -1) {
				nameTemplate = nameTemplate.substring(pos + 1);
			}
			pos = nameTemplate.lastIndexOf('\\');
			if (pos != -1) {
				nameTemplate = nameTemplate.substring(pos + 1);
			}
		}

		if (StringUtils.isBlank(nameTemplate)) {
			prefix = "hostsession";
			suffix = ".tmp";
		} else {
			prefix = FilenameUtils.getBaseName(nameTemplate);
			suffix = "." + FilenameUtils.getExtension(nameTemplate);
		}

		return getTempFile(prefix, suffix);
	}

	public HostFile copyToTemporaryFile(File localFile) throws RuntimeIOException {
		HostFile t = getTempFile(localFile.getName());
		t.put(localFile);
		return t;
	}

	public HostFile copyToTemporaryFile(Resource resource) throws RuntimeIOException {
		HostFile t = getTempFile(resource.getFilename());
		t.put(resource);
		return t;
	}

	private static Logger logger = Logger.getLogger(AbstractHostSession.class);

}
