/*
 * Copyright (c) 2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of Xebialabs Software and Documentation is subject to the Personal
 * License Agreement.
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * ‚ÄúDocumentation‚Äù means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the Xebialabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to  (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.reflect;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang.ClassUtils;

import com.xebialabs.deployit.ConfigurationItem;
import com.xebialabs.deployit.ConfigurationItemDescription;
import com.xebialabs.deployit.ConfigurationItemLabel;
import com.xebialabs.deployit.ConfigurationItemPermissionScheme;
import com.xebialabs.deployit.ci.security.PermissionScheme;

/**
 * Describes a {@link ConfigurationItem}.
 */
@SuppressWarnings("serial")
public class ConfigurationItemDescriptor implements Serializable, Comparable<ConfigurationItemDescriptor>, ConfigurationItemPropertyHolder {
	/*
	 * N.B.: Default constructor should be there for BlazeDS serialization (AS -> Java) to work.
	 * 
	 * N.B.: Getters and setters should both be there for BlazeDS serialization (Java -> AS) to work
	 */

	private String type;
	private transient Class<?> typeClass;
	private String simpleName;

	private Set<String> interfaces = new HashSet<String>();

	private ArrayList<String> superClasses = new ArrayList<String>();

	private String description;
	private ConfigurationItemPropertyDescriptor[] propertyDescriptors = new ConfigurationItemPropertyDescriptor[0];

	private String labelPropertyName;
	private transient Field labelPropertyField;

	private String descriptionPropertyName;
	private transient Field descriptionPropertyField;

	private String permissionSchemePropertyName;
	private transient Field permissionSchemePropertyField;

	private String defaultDescription;
	private String defaultPermissionSchemeLabel;

	private String category;
	private boolean expandable = false;

	private String[] sourceForMappingTypes = new String[0];
	private String[] targetForMappingTypes = new String[0];

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public ConfigurationItemDescriptor() {
	}

	/**
	 * Constructs a <tt>ConfigurationItemDescriptor</tt>. Do not invoke!
	 */
	public ConfigurationItemDescriptor(Class<?> typeClass, Map<String, String> configuredDefaults) {
		initTypeInfo(typeClass);

		ConfigurationItem ciAnnotation = typeClass.getAnnotation(ConfigurationItem.class);
		initDescription(ciAnnotation);

		String defaultValueKeyPrefix = typeClass.getSimpleName().toLowerCase() + ".";
		initProperties(typeClass, configuredDefaults, defaultValueKeyPrefix);
		initBuiltinProperties(typeClass);

		initDefaults(configuredDefaults, defaultValueKeyPrefix);
		initCategories(ciAnnotation);
		initValidRelations(ciAnnotation);
	}

	private void initValidRelations(ConfigurationItem ciAnnotation) {
		setSourceForMappingTypes(classToStringArray(ciAnnotation.sourceForMappingTypes()));
		setTargetForMappingTypes(classToStringArray(ciAnnotation.targetForMappingTypes()));
	}

	private void initBuiltinProperties(Class<?> typeClass) {
		initLabelProperty(typeClass);
		initDescriptionProperty(typeClass);
		initPermissionSchemeProperty(typeClass);
	}

	private void initLabelProperty(Class<?> typeClass) {
		List<Field> allLabelProperties = ConfigurationItemPropertyDescriptor.getAllAnnotatedProperties(typeClass, ConfigurationItemLabel.class);
		if (allLabelProperties.size() == 1) {
			setLabelPropertyField(allLabelProperties.get(0));
		} else if (allLabelProperties.size() > 1) {
			throw new IllegalArgumentException("CI type " + type + " defines more than one property with the @ConfigurationItemLabel annotation");
		}
	}

	private void initDescriptionProperty(Class<?> typeClass) {
		List<Field> allDescriptionProperties = ConfigurationItemPropertyDescriptor.getAllAnnotatedProperties(typeClass, ConfigurationItemDescription.class);
		if (allDescriptionProperties.size() == 1) {
			setDescriptionPropertyField(allDescriptionProperties.get(0));
		} else if (allDescriptionProperties.size() > 1) {
			throw new IllegalArgumentException("CI type " + type + " defines more than one property with the @ConfigurationItemDescription annotation");
		}
	}

	private void initPermissionSchemeProperty(Class<? extends Object> typeClass) {
		List<Field> allPermissionSchemeProperties = ConfigurationItemPropertyDescriptor.getAllAnnotatedProperties(typeClass,
				ConfigurationItemPermissionScheme.class);
		if (allPermissionSchemeProperties.size() == 1) {
			setPermissionSchemePropertyField(allPermissionSchemeProperties.get(0));
		} else if (allPermissionSchemeProperties.size() > 1) {
			throw new IllegalArgumentException("CI type " + type + " defines more than one property with the @ConfigurationItemPermissionScheme annotation");
		}
	}

	@SuppressWarnings("unchecked")
	private void initTypeInfo(Class<? extends Object> typeClass) {
		this.typeClass = typeClass;
		this.type = typeClass.getName();
		this.simpleName = typeClass.getSimpleName();

		for (Class each : (List<Class>) ClassUtils.getAllInterfaces(typeClass)) {
			interfaces.add(each.getName());
		}

		List<Class> allSuperclasses = ClassUtils.getAllSuperclasses(typeClass);
		for (Class each : allSuperclasses) {
			if (!each.toString().equalsIgnoreCase(java.lang.Object.class.toString())) {
				superClasses.add(each.getName());
			}
		}

	}

	private void initDescription(ConfigurationItem ciAnnotation) {
		this.description = ciAnnotation.description();
	}

	private void initProperties(Class<? extends Object> typeClass, Map<String, String> configuredDefaults, String defaultValueKeyPrefix) {
		this.propertyDescriptors = ConfigurationItemPropertyDescriptor.getConfigurationItemPropertyDescriptorsForTopLevelClass(this, typeClass,
				configuredDefaults, defaultValueKeyPrefix);
	}

	private void initDefaults(Map<String, String> configuredDefaults, String defaultValueKeyPrefix) {
		this.defaultDescription = configuredDefaults.get(defaultValueKeyPrefix + "description");
		this.defaultPermissionSchemeLabel = configuredDefaults.get(defaultValueKeyPrefix + "permissionscheme");
	}

	private void initCategories(ConfigurationItem ciAnnotation) {
		this.category = ciAnnotation.category();
		this.expandable = ciAnnotation.expandable();
	}

	/**
	 * Gets the type this descriptor describes.
	 * 
	 * @return the full class name of the type, e.g. <code>com.xebia.ad.websphere.server.WebSphereServer<code>.
	 */
	public String getType() {
		return type;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setType(String type) {
		this.type = type;
	}

	/**
	 * Return the class object of the type this descriptor describes.
	 * 
	 * @return the class object of the type.
	 */
	public Class<? extends Object> getTypeClass() {
		if (typeClass == null) {
			try {
				typeClass = (Class<? extends Object>) Class.forName(type);
			} catch (ClassNotFoundException exc) {
				throw new RuntimeException("Cannot load class " + type, exc);
			}
		}
		return typeClass;
	}

	/**
	 * Gets the pretty name of the type this descriptor describes.
	 * 
	 * @return the pretty name of the type, e.g. <code>WebSphereServer</code>.
	 */
	public String getSimpleName() {
		return simpleName;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setSimpleName(String simpleName) {
		this.simpleName = simpleName;
	}

	/**
	 * Creates a new instance of the configuration item type described by this <tt>ConfigurationItemDescriptor</tt>.
	 * 
	 * @return the new instance.
	 */
	public Object newInstance() {
		try {
			return typeClass.newInstance();
		} catch (Exception exception) {
			throw new AssertionError("Unable to instantiate type " + type + ": " + exception.getMessage());
		}
	}

	/**
	 * Gets the descriptors of the properties of the configuration item type described by this
	 * <tt>ConfigurationItem=Descriptor</tt>.
	 * 
	 * @return an array of {@link ConfigurationItemPropertyDescriptor}s.
	 */
	public ConfigurationItemPropertyDescriptor[] getPropertyDescriptors() {
		return propertyDescriptors;
	}

	public ConfigurationItemPropertyDescriptor getPropertyDescriptor(String propertyName) {
		for (ConfigurationItemPropertyDescriptor each : propertyDescriptors) {
			if (each.getName().equalsIgnoreCase(propertyName)) {
				return each;
			}
		}
		return null;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setPropertyDescriptors(ConfigurationItemPropertyDescriptor[] propertyDescriptors) {
		this.propertyDescriptors = propertyDescriptors;
	}

	/**
	 * Gets the name of the field that has been annotated with {@link ConfigurationItemLabel} to hold the label.
	 * 
	 * @return the name of the field that holds the label or <tt>null</tt> if there is no such field.
	 */
	public String getLabelPropertyName() {
		return labelPropertyName;
	}

	/**
	 * Gets the {@link Field} that has been annotated with {@link ConfigurationItemLabel} to hold the label.
	 * 
	 * @return the field that holds the label or <tt>null</tt> if there is no such field.
	 */
	public Field getLabelPropertyField() {
		return labelPropertyField;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setLabelPropertyName(String labelPropertyName) {
		this.labelPropertyName = labelPropertyName;
	}

	private void setLabelPropertyField(Field labelPropertyField) {
		if (!labelPropertyField.getType().equals(String.class)) {
			throw new IllegalArgumentException("Property " + labelPropertyField.getName() + " of class " + labelPropertyField.getDeclaringClass().getName()
					+ " is annotated with the @ConfigurationItemLabel annotation but is not a String");
		}

		labelPropertyField.setAccessible(true);
		this.labelPropertyField = labelPropertyField;
		this.labelPropertyName = labelPropertyField.getName();
	}

	/**
	 * Gets a label from a configuration item.
	 * 
	 * @param configurationItem
	 *            the configuration item to get the label from.
	 * @return the label or <tt>null</tt> if a label has not been set or if there is no {@link #getLabelPropertyField()
	 *         label property}.
	 */
	public String getLabelValueFromConfigurationItem(Object configurationItem) {
		configurationItem = ConfigurationItemReflectionUtils.getRealConfigurationItem(configurationItem);
		return (String) getFieldValueOrReturnNull(labelPropertyField, configurationItem);
	}

	/**
	 * Sets a label on a configuration item.
	 * 
	 * @param configurationItem
	 *            the configuration item to set the label on.
	 * @param labelValue
	 *            the label to set
	 */
	public void setLabelValueInConfigurationItem(Object configurationItem, String labelValue) {
		configurationItem = ConfigurationItemReflectionUtils.getRealConfigurationItem(configurationItem);
		setFieldValue(labelPropertyField, configurationItem, labelValue);
	}

	/**
	 * Gets the name of the field that has been annotated with {@link ConfigurationItemDescription} to hold the
	 * description.
	 * 
	 * @return the name of the field that holds the description or <tt>null</tt> if there is no such field.
	 */
	public String getDescriptionPropertyName() {
		return descriptionPropertyName;
	}

	/**
	 * Gets the {@link Field} that has been annotated with {@link ConfigurationItemDescription} to hold the description.
	 * 
	 * @return the field that holds the description or <tt>null</tt> if there is no such field.
	 */
	public Field getDescriptionPropertyField() {
		return descriptionPropertyField;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setDescriptionPropertyName(String descriptionPropertyName) {
		this.descriptionPropertyName = descriptionPropertyName;
	}

	private void setDescriptionPropertyField(Field descriptionPropertyField) {
		if (!descriptionPropertyField.getType().equals(String.class)) {
			throw new IllegalArgumentException("Property " + descriptionPropertyField.getName() + " of class "
					+ descriptionPropertyField.getDeclaringClass().getName()
					+ " is annotated with the @ConfigurationItemDescription annotation but is not a String");
		}

		descriptionPropertyField.setAccessible(true);
		this.descriptionPropertyField = descriptionPropertyField;
		this.descriptionPropertyName = descriptionPropertyField.getName();
	}

	/**
	 * Gets a description from a configuration item.
	 * 
	 * @param configurationItem
	 *            the configuration item to get the description from.
	 * @return the label or <tt>null</tt> if a description has not been set or if there is no
	 *         {@link #getDescriptionPropertyField() description property}.
	 */
	public String getDescriptionValueFromConfigurationItem(Object configurationItem) {
		configurationItem = ConfigurationItemReflectionUtils.getRealConfigurationItem(configurationItem);
		return (String) getFieldValueOrReturnNull(descriptionPropertyField, configurationItem);
	}

	/**
	 * Sets a description on a configuration item.
	 * 
	 * @param configurationItem
	 *            the configuration item to set the label on.
	 * @param descriptionValue
	 *            the description to set
	 */
	public void setDescriptionValueInConfigurationItem(Object configurationItem, String descriptionValue) {
		configurationItem = ConfigurationItemReflectionUtils.getRealConfigurationItem(configurationItem);
		setFieldValue(descriptionPropertyField, configurationItem, descriptionValue);
	}

	/**
	 * Gets the default description for newly created configuration items of this type.
	 * 
	 * @return the default description.
	 */
	public String getDefaultDescription() {
		return defaultDescription;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setDefaultDescription(String defaultDescription) {
		this.defaultDescription = defaultDescription;
	}

	/**
	 * Gets the name of the field that has been annotated with {@link ConfigurationItemPermissionScheme} to hold the
	 * {@link PermissionScheme}.
	 * 
	 * @return the name of the field that holds the permission scheme or <tt>null</tt> if there is no such field.
	 */
	public String getPermissionSchemePropertyName() {
		return permissionSchemePropertyName;
	}

	/**
	 * Gets the {@link Field} that has been annotated with {@link ConfigurationItemPermissionScheme} to hold the
	 * {@link PermissionScheme}.
	 * 
	 * @return the field that holds the permission scheme or <tt>null</tt> if there is no such field.
	 */
	public Field getPermissionSchemePropertyField() {
		return permissionSchemePropertyField;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setPermissionSchemePropertyName(String permissionSchemePropertyName) {
		this.permissionSchemePropertyName = permissionSchemePropertyName;
	}

	private void setPermissionSchemePropertyField(Field permissionSchemePropertyField) {
		if (!permissionSchemePropertyField.getType().equals(PermissionScheme.class)) {
			throw new IllegalArgumentException("Property " + permissionSchemePropertyField.getName() + " of class "
					+ descriptionPropertyField.getDeclaringClass().getName()
					+ " is annotated with the @ConfigurationItemPermissionScheme annotation but is not a String");
		}

		permissionSchemePropertyField.setAccessible(true);
		this.permissionSchemePropertyField = permissionSchemePropertyField;
		this.permissionSchemePropertyName = permissionSchemePropertyField.getName();
	}

	/**
	 * Gets a permission scheme from a configuration item.
	 * 
	 * @param configurationItem
	 *            the configuration item to get the description from.
	 * @return the permission scheme or <tt>null</tt> if a permission scheme has not been set or if there is no
	 *         {@link #getPermissionSchemePropertyField() permission scheme property}.
	 */
	public PermissionScheme getPermissionSchemeValueFromConfigurationItem(Object configurationItem) {
		configurationItem = ConfigurationItemReflectionUtils.getRealConfigurationItem(configurationItem);
		return (PermissionScheme) getFieldValueOrReturnNull(permissionSchemePropertyField, configurationItem);
	}

	/**
	 * Sets a {@link PermissionScheme} on a configuration item.
	 * 
	 * @param configurationItem
	 *            the configuration item to set the permission scheme on.
	 * @param permissionSchemeValue
	 *            the permission scheme to set
	 */
	public void setPermissionSchemeValueInConfigurationItem(Object configurationItem, PermissionScheme permissionSchemeValue) {
		configurationItem = ConfigurationItemReflectionUtils.getRealConfigurationItem(configurationItem);
		setFieldValue(permissionSchemePropertyField, configurationItem, permissionSchemeValue);
	}

	/**
	 * Gets the label of the default permission scheme for newly created configuration items of this type.
	 * 
	 * @return the label of the default permission scheme.
	 */
	public String getDefaultPermissionSchemeLabel() {
		return defaultPermissionSchemeLabel;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setDefaultPermissionSchemeLabel(String defaultPermissionSchemeLabel) {
		this.defaultPermissionSchemeLabel = defaultPermissionSchemeLabel;
	}

	private Object getFieldValueOrReturnNull(Field propertyField, Object theObject) {
		if (propertyField != null) {
			propertyField.setAccessible(true);
			try {
				return propertyField.get(theObject);
			} catch (IllegalArgumentException e) {
				throw new RuntimeException("Unable to access field " + propertyField.getName() + " for CI " + theObject.getClass().getName(), e);
			} catch (IllegalAccessException e) {
				throw new RuntimeException("Unable to access field " + propertyField.getName() + " for CI " + theObject.getClass().getName(), e);
			}
		} else {
			return null;
		}
	}

	private void setFieldValue(Field propertyField, Object theObject, Object theValue) {
		if (propertyField != null) {
			propertyField.setAccessible(true);
			try {
				propertyField.set(theObject, theValue);
			} catch (IllegalArgumentException e) {
				throw new RuntimeException("Unable to access field " + propertyField.getName() + " for CI " + theObject.getClass().getName(), e);
			} catch (IllegalAccessException e) {
				throw new RuntimeException("Unable to access field " + propertyField.getName() + " for CI " + theObject.getClass().getName(), e);
			}
		}
	}

	/**
	 * Gets the names of the interfaces implemented by this configuration item.
	 * 
	 * @return a set containing the names of the interfaces.
	 */
	public Set<String> getInterfaces() {
		return interfaces;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setInterfaces(Set<String> interfaces) {
		this.interfaces = interfaces;
	}

	/**
	 * Gets the names of the superclasses of this configuration item.
	 * <p>
	 * N.B.: The return type needs to be specified as {@link ArrayList} instead of {@link List} for BlazeDS.
	 * 
	 * @return a list of the names of the superclasses.
	 */
	public ArrayList<String> getSuperClasses() {
		return superClasses;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setSuperClasses(ArrayList<String> superClasses) {
		this.superClasses = superClasses;
	}

	/**
	 * Gets the description of this configuration item type.
	 * 
	 * @return the description.
	 */
	public String getDescription() {
		return description;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setDescription(String description) {
		this.description = description;
	}

	/**
	 * Gets the category of this configuration item type.
	 * 
	 * @return the category.
	 */
	public String getCategory() {
		return this.category;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setCategory(String category) {
		this.category = category;
	}

	/**
	 * Returns whether this configuration item type should be shown as expandable in the UI tree.
	 * 
	 * @return <tt>true</tt> iff this configuration item type should be shown as expandable in the UI tree.
	 */
	public boolean isExpandable() {
		return expandable;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setExpandable(boolean expandable) {
		this.expandable = expandable;
	}

	/**
	 * Gets the source types for this configuration item type if this configuration item type is a Mapping.
	 */
	public String[] getSourceForMappingTypes() {
		return sourceForMappingTypes;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setSourceForMappingTypes(String[] sourceForMappingTypes) {
		this.sourceForMappingTypes = sourceForMappingTypes;
	}

	/**
	 * Gets the target types for this configuration item type if this configuration item type is a Mapping.
	 */
	public String[] getTargetForMappingTypes() {
		return targetForMappingTypes;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setTargetForMappingTypes(String[] targetForMappingTypes) {
		this.targetForMappingTypes = targetForMappingTypes;
	}

	/**
	 * Compares this <tt>ConfigurationItemDescriptor</tt> to another and return -1, 0 or 1 depending on the difference.
	 */
	public int compareTo(ConfigurationItemDescriptor descriptor) {
		return this.simpleName.compareTo(descriptor.simpleName);
	}

	@Override
	public String toString() {
		return "ConfigurationItemTypeDescriptor of " + getType();
	}

	private static String[] classToStringArray(Class<?>[] classArray) {
		String[] stringArray = new String[classArray.length];
		int i = 0;
		for (Class<?> clazz : classArray) {
			stringArray[i] = clazz.getName();
			i++;
		}
		return stringArray;
	}

}