package com.xebialabs.deployit.hostsession.cifs;

import static com.google.common.base.Preconditions.checkNotNull;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collections;
import java.util.Map;
import java.util.Random;

import jcifs.smb.SmbFile;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.net.telnet.InvalidTelnetOptionException;
import org.apache.commons.net.telnet.TelnetClient;
import org.apache.commons.net.telnet.WindowSizeOptionHandler;
import org.apache.log4j.Logger;

import com.xebialabs.deployit.ci.OperatingSystemFamily;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.hostsession.CommandExecution;
import com.xebialabs.deployit.hostsession.CommandExecutionCallbackHandler;
import com.xebialabs.deployit.hostsession.HostFile;
import com.xebialabs.deployit.hostsession.HostSession;
import com.xebialabs.deployit.hostsession.common.AbstractHostSession;

/**
 * <ul>
 * <li>Windows Telnet Service in stream mode:<br/>
 * <tt>&gt; tlntadmn config mode=stream</tt></li>
 * <li>Shares with names like C$ are available for all drives you access.</li>
 * <li>Not tested with domain accounts.</li>
 * </ul>
 */
public class CifsTelnetHostSession extends AbstractHostSession implements HostSession {

	private String address;

	private String username;

	private String password;

	private static final String DETECTABLE_WINDOWS_PROMPT = "WINDOWS4DEPLOYIT ";

	private static final String ERRORLEVEL_PREAMBLE = "ERRORLEVEL-PREAMBLE";

	private static final String ERRORLEVEL_POSTAMBLE = "ERRORLEVEL-POSTAMBLE";

	/**
	 * The exitcode returned when the errorlevel of the Windows command could not be determined.
	 */
	public static final int EXITCODE_CANNOT_DETERMINE_ERRORLEVEL = -999999;

	public CifsTelnetHostSession(OperatingSystemFamily osFamily, String temporaryDirectoryPath, String address, int port, String username, String password) {
		super(osFamily, temporaryDirectoryPath);
		this.address = address;
		this.username = username;
		this.password = password;
	}

	@SuppressWarnings("unchecked")
	public int execute(CommandExecutionCallbackHandler handler, String... commandLine) throws RuntimeIOException {
		return execute(handler, Collections.EMPTY_MAP, commandLine);
	}

	public int execute(CommandExecutionCallbackHandler handler, Map<String, String> inputResponse, String... commandLine) {
		String command = encodeCommandLine(false, commandLine);
		String commandWithHiddenPassword = encodeCommandLine(true, commandLine);

		try {
			TelnetClient tc = new TelnetClient();
			tc.addOptionHandler(new WindowSizeOptionHandler(299, 25, true, false, true, false));
			tc.connect(address);
			try {
				InputStream stdout = tc.getInputStream();
				OutputStream stdin = tc.getOutputStream();
				StringBuffer outputBuf = new StringBuffer();

				waitFor(handler, stdout, outputBuf, "ogin:");
				send(handler, stdin, username);

				waitFor(handler, stdout, outputBuf, "assword:");
				send(handler, stdin, password);

				waitFor(handler, stdout, outputBuf, ">");
				send(handler, stdin, "PROMPT " + DETECTABLE_WINDOWS_PROMPT);
				// We must wait for the prompt twice; the first time is an echo of the PROMPT command,
				// the second is the actual prompt
				waitFor(handler, stdout, outputBuf, DETECTABLE_WINDOWS_PROMPT);
				waitFor(handler, stdout, outputBuf, DETECTABLE_WINDOWS_PROMPT);

				send(handler, stdin, command);
				waitFor(handler, stdout, outputBuf, DETECTABLE_WINDOWS_PROMPT);

				send(handler, stdin, "ECHO \"" + ERRORLEVEL_PREAMBLE + "%errorlevel%" + ERRORLEVEL_POSTAMBLE);
				waitFor(handler, stdout, outputBuf, ERRORLEVEL_POSTAMBLE);
				waitFor(handler, stdout, outputBuf, ERRORLEVEL_POSTAMBLE);
				int preamblePos = outputBuf.indexOf(ERRORLEVEL_PREAMBLE);
				int postamblePos = outputBuf.indexOf(ERRORLEVEL_POSTAMBLE);
				if (preamblePos >= 0 && postamblePos >= 0) {
					String errorlevelString = outputBuf.substring(preamblePos + ERRORLEVEL_PREAMBLE.length(), postamblePos);
					if (logger.isDebugEnabled())
						logger.debug("Errorlevel string found: " + errorlevelString);

					try {
						return Integer.parseInt(errorlevelString);
					} catch (NumberFormatException exc) {
						logger.error("Cannot parse errorlevel in Windows output: " + outputBuf);
						return EXITCODE_CANNOT_DETERMINE_ERRORLEVEL;
					}
				} else {
					logger.error("Cannot find errorlevel in Windows output: " + outputBuf);
					return EXITCODE_CANNOT_DETERMINE_ERRORLEVEL;
				}
			} finally {
				tc.disconnect();
			}
		} catch (InvalidTelnetOptionException exc) {
			throw new RuntimeIOException("Cannot execute command " + commandWithHiddenPassword + " on " + address, exc);
		} catch (IOException exc) {
			throw new RuntimeIOException("Cannot execute command " + commandWithHiddenPassword + " on " + address, exc);
		}
	}

	private void waitFor(CommandExecutionCallbackHandler handler, InputStream stdout, StringBuffer outputBuf, String expectedString) throws IOException {
		boolean lastCharWasCr = false;
		boolean lastCharWasEsc = false;
		for (;;) {
			int c = stdout.read();
			if (c == -1) {
				throw new IOException("End of stream reached");
			}

			handler.handleOutput((char) c);
			switch ((char) c) {
			case '\r':
				handler.handleOutputLine(outputBuf.toString());
				outputBuf.delete(0, outputBuf.length());
				break;
			case '\n':
				if (!lastCharWasCr) {
					handler.handleOutputLine(outputBuf.toString());
					outputBuf.delete(0, outputBuf.length());
				}
				break;
			case '[':
				if (lastCharWasEsc) {
					throw new RuntimeIOException(
							"VT100/ANSI escape sequence found in output stream. Please configure the Windows Telnet server to use stream mode.");
				}
			default:
				outputBuf.append((char) c);
				break;
			}
			lastCharWasCr = (c == '\r');
			lastCharWasEsc = (c == 27);

			if (outputBuf.length() >= expectedString.length()) {
				String s = outputBuf.substring(outputBuf.length() - expectedString.length(), outputBuf.length());
				if (s.equals(expectedString)) {
					if (logger.isDebugEnabled()) {
						logger.debug("Expected string \"" + expectedString + "\" found in Windows Telnet output");
					}
					return;
				}
			}
		}
	}

	private void send(CommandExecutionCallbackHandler handler, OutputStream stdin, String lineToSend) throws IOException {
		byte[] bytesToSend = (lineToSend + "\r\n").getBytes();
		stdin.write(bytesToSend);
		stdin.flush();
	}

	public CommandExecution startExecute(String... commandLine) {
		return null;
	}

	public HostFile getFile(String hostPath) throws RuntimeIOException {
		try {
			SmbFile smbFile = new SmbFile(encodeAsSmbUrl(hostPath));
			return new CifsHostFile(this, smbFile);
		} catch (IOException exc) {
			throw new RuntimeIOException(exc);
		}
	}

	public HostFile getFile(HostFile parent, String child) throws RuntimeIOException {
		return getFile(parent.getPath() + getHostOperatingSystem().getFileSeparator() + child.replace('\\', '/'));
	}

	public HostFile getTempFile(String prefix, String suffix) throws RuntimeIOException {
		checkNotNull(prefix);
		if (suffix == null) {
			suffix = ".tmp";
		}

		Random r = new Random();
		String infix = "";
		for (int i = 0; i < AbstractHostSession.MAX_TEMP_RETRIES; i++) {
			HostFile f = getFile(getTemporaryDirectory().getPath() + getHostOperatingSystem().getFileSeparator() + prefix + infix + suffix);
			if (!f.exists()) {
				if (logger.isDebugEnabled())
					logger.debug("Created temporary file " + f);

				return f;
			}
			infix = "-" + Long.toString(Math.abs(r.nextLong()));
		}
		throw new RuntimeIOException("Cannot generate a unique temporary file name on " + this);
	}

	private String encodeAsSmbUrl(String hostPath) {
		StringBuffer smbUrl = new StringBuffer();
		smbUrl.append("smb://");
		smbUrl.append(username);
		smbUrl.append(":");
		smbUrl.append(password);
		smbUrl.append("@");
		smbUrl.append(address);
		smbUrl.append("/");

		if (hostPath.length() < 2) {
			throw new RuntimeIOException("Host path \"" + hostPath + "\" is too short");
		}

		if (hostPath.charAt(1) != ':') {
			throw new RuntimeIOException("Host path \"" + hostPath + "\" does not have a colon (:) as its second character");
		}
		smbUrl.append(hostPath.charAt(0));
		smbUrl.append("$/");
		if (hostPath.length() >= 3) {
			if (hostPath.charAt(2) != '\\') {
				throw new RuntimeIOException("Host path \"" + hostPath + "\" does not have a backslash (\\) as its third character");
			}
			smbUrl.append(hostPath.substring(3).replace('\\', '/'));
		}

		if (logger.isDebugEnabled())
			logger.debug("Encoded Windows host path \"" + hostPath + "\" to SMB URL \"" + smbUrl.toString() + "\"");
		return smbUrl.toString();
	}

	private static String encodeCommandLine(boolean hidePassword, String... commandLine) {
		if (commandLine == null || commandLine.length == 0) {
			throw new IllegalStateException("Cannot execute an empty command line");
		}

		StringBuilder sb = new StringBuilder();
		boolean passwordKeywordSeen = false;
		for (int i = 0; i < commandLine.length; i++) {
			if (i != 0) {
				sb.append(' ');
			}
			if (commandLine[i] == null) {
				sb.append("null");
			} else if (commandLine[i].length() == 0) {
				sb.append("\" \"");
			} else {
				if (passwordKeywordSeen && hidePassword) {
					for (int j = 0; j < commandLine[i].length(); j++) {
						sb.append("*");
					}
				} else {
					for (int j = 0; j < commandLine[i].length(); j++) {
						char c = commandLine[i].charAt(j);
						// if (" '\";()${}".indexOf(c) != -1) {
						// sb.append('\\');
						// }
						sb.append(c);
					}
				}
				// catch 'password' or '-password'
				passwordKeywordSeen = StringUtils.endsWithIgnoreCase(commandLine[i], "password");
			}
		}
		return sb.toString();
	}

	public String toString() {
		return username + "@" + address;
	}

	private static Logger logger = Logger.getLogger(CifsTelnetHostSession.class);

}
