/*
 * Copyright (c) 2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to  (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.reflect;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.lang.BooleanUtils;

import com.xebialabs.deployit.ConfigurationItemProperty;
import com.xebialabs.deployit.ci.artifact.DeployableArtifact;
import com.xebialabs.deployit.ci.security.PermissionScheme;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.util.ExtendedFileUtils;

/**
 * Contains a number of configuration item reflection utility methods.
 */
public class ConfigurationItemReflectionUtils {

	/**
	 * Returns the real configuration item if this is a {@link ConfigurationItemProxy}.
	 * 
	 * @param <T>
	 *            the type of the configuration item
	 * @param possibleCiProxy
	 *            the object that is possibly a configuration item proxy.
	 * @return the backing object if <tt>possibleCiProxy</tt> is a proxy, its value if it is not a proxy.
	 */
	@SuppressWarnings("unchecked")
	public static <T> T getRealConfigurationItem(T possibleCiProxy) {
		if (possibleCiProxy instanceof ConfigurationItemProxy) {
			return (T) ((ConfigurationItemProxy) possibleCiProxy).getBackingObject();
		} else {
			return possibleCiProxy;
		}
	}

	/**
	 * Compares two configuration items and determines whether they are "similar". Two configuration items are "similar"
	 * if their {@link ConfigurationItemProperty#identifying() identifying} properties have the same value.
	 * 
	 * @param ci1
	 *            the first configration item to be compared
	 * @param ci2
	 *            the second configration item to be compared
	 * @return <tt>true</tt> if and only if the configuration item are considered "similar".
	 */
	public static boolean isSimilar(Serializable ci1, Serializable ci2) {
		return compareProperties(ci1, ci2, true);
	}

	/**
	 * Compares two configuration items and determines whether they are "identical". Two configuration items are
	 * "identical" if <strong>all</strong> their properties have the same value.
	 * 
	 * @param ci1
	 *            the first configration item to be compared
	 * @param ci2
	 *            the second configration item to be compared
	 * @return <tt>true</tt> if and only if the configuration item are considered "identical".
	 */
	public static boolean isIdentical(Serializable ci1, Serializable ci2) {
		return compareProperties(ci1, ci2, false);
	}

	private static boolean compareProperties(Serializable ci1, Serializable ci2, boolean compareIdentifyingPropertiesOnly) {
		if (ci1 == null || ci2 == null) {
			return false;
		}

		ci1 = getRealConfigurationItem(ci1);
		ci2 = getRealConfigurationItem(ci2);

		ConfigurationItemDescriptor desc = ConfigurationItemDescriptorFactory.getDescriptor(ci1);
		if (desc == null) {
			throw new IllegalArgumentException("First argument is not a CI, it is of class " + ci1.getClass().getName());
		}

		ConfigurationItemDescriptor desc2 = ConfigurationItemDescriptorFactory.getDescriptor(ci2);
		if (desc2 == null) {
			throw new IllegalArgumentException("Second argument is not a CI, it is of class " + ci2.getClass().getName());
		}

		if (desc != desc2) {
			return false;
		}

		return compareProperties(desc.getPropertyDescriptors(), ci1, ci2, compareIdentifyingPropertiesOnly);
	}

	private static boolean compareProperties(ConfigurationItemPropertyDescriptor[] propertyDescriptors, Serializable ci1, Serializable ci2,
			boolean compareIdentifyingPropertiesOnly) {
		for (ConfigurationItemPropertyDescriptor pd : propertyDescriptors) {
			if (compareIdentifyingPropertiesOnly && !pd.isIdentifying()) {
				continue;
			}

			Object val1 = pd.getPropertyValueFromConfigurationItem(ci1);
			Object val2 = pd.getPropertyValueFromConfigurationItem(ci2);

			if (!comparePropertyValues(pd, val1, val2)) {
				return false;
			}
		}
		return true;
	}

	private static boolean comparePropertyValues(ConfigurationItemPropertyDescriptor pd, Object val1, Object val2) {
		if (val1 == null) {
			if (val2 != null) {
				return false;
			}
		} else if (val2 == null) {
			return false;
		} else {
			if (pd.getPropertyField().getName().equals(DeployableArtifact.NAME_OF_CONTENTS_FIELD)
					&& pd.getPropertyField().getDeclaringClass() == DeployableArtifact.class) {
				return compareDeployableArtifactContents(val1, val2);
			} else if (pd.getType() == ConfigurationItemPropertyType.LIST_OF_OBJECTS) {
				return compareListOfObjectsProperties(pd, val1, val2);
			} else if (pd.getType() == ConfigurationItemPropertyType.CI || pd.getType() == ConfigurationItemPropertyType.SET_OF_CIS) {
				return true;
			} else {
				if (!val1.equals(val2)) {
					return false;
				}
			}
		}
		return true;
	}

	private static boolean compareListOfObjectsProperties(ConfigurationItemPropertyDescriptor pd, Object val1, Object val2) {
		List<?> l1 = (List<?>) val1;
		List<?> l2 = (List<?>) val2;
		if (l1.size() != l2.size()) {
			return false;
		}
		for (int i = 0; i < l1.size(); i++) {
			Serializable li1 = (Serializable) l1.get(i);
			Serializable li2 = (Serializable) l1.get(i);
			if (!compareProperties(pd.getListObjectPropertyDescriptors(), li1, li2, false)) {
				return false;
			}
		}
		return true;
	}

	private static boolean compareDeployableArtifactContents(Object val1, Object val2) {
		if (val1 == null || !(val1 instanceof String) || val2 == null || !(val2 instanceof String)) {
			return false;
		}
		File file1 = new File((String) val1);
		File file2 = new File((String) val2);
		try {
			return ExtendedFileUtils.contentEquals(file1, file2);
		} catch (IOException exc) {
			throw new RuntimeIOException("Cannot compare files " + file1 + " and " + file2, exc);
		}
	}

	/**
	 * Overrides named properties in the given CI with the corresponding values in the {@code propertyOverrides} map.
	 * <strong>N.B.:</strong> Modifes the CI passed in!
	 */
	public static Serializable overrideProperties(Serializable ci, Map<String, String> propertyOverrides) {
		ci = getRealConfigurationItem(ci);

		ConfigurationItemDescriptor ciDescriptor = ConfigurationItemDescriptorFactory.getDescriptor(checkNotNull(ci));
		if (ciDescriptor == null) {
			throw new IllegalArgumentException("Unknown CI type " + ci.getClass());
		}

		Serializable copyOfCi = (Serializable) ciDescriptor.newInstance();

		// Copy original properties
		for (ConfigurationItemPropertyDescriptor spd : ciDescriptor.getPropertyDescriptors()) {
			Object value = spd.getPropertyValueFromConfigurationItem(ci);
			spd.setPropertyValueInConfigurationItem(copyOfCi, value);
		}

		// Overlay properties
		for (Entry<String, String> propertyOverride : checkNotNull(propertyOverrides).entrySet()) {
			ConfigurationItemPropertyDescriptor overriddenPropertyDescriptor = ciDescriptor.getPropertyDescriptor(propertyOverride.getKey());
			checkArgument(overriddenPropertyDescriptor != null);
			checkArgument(!overriddenPropertyDescriptor.isIdentifying(), "Unable to override an identifying property: "
					+ overriddenPropertyDescriptor.getName());
			overriddenPropertyDescriptor.setPropertyValueInConfigurationItem(copyOfCi, ConfigurationItemReflectionUtils.toPropertyValue(
					overriddenPropertyDescriptor, propertyOverride.getValue()));
		}
		ConfigurationItemReflectionUtils.copyIntoLabelDescriptionAndPermissionScheme(ci, copyOfCi, ciDescriptor);
		return copyOfCi;
	}

	static void copyIntoLabelDescriptionAndPermissionScheme(Serializable sourceCi, Serializable copyOfCi, ConfigurationItemDescriptor ciDescriptor) {
		// Label
		String sourceLabelValue = ciDescriptor.getLabelValueFromConfigurationItem(sourceCi);
		if (sourceLabelValue != null) {
			ciDescriptor.setLabelValueInConfigurationItem(copyOfCi, sourceLabelValue.toString());
		}

		// Description
		String dValue = ciDescriptor.getDescriptionValueFromConfigurationItem(sourceCi);
		if (dValue != null) {
			ciDescriptor.setDescriptionValueInConfigurationItem(copyOfCi, dValue);
		}

		// PermissionScheme
		PermissionScheme psValue = ciDescriptor.getPermissionSchemeValueFromConfigurationItem(sourceCi);
		if (psValue != null) {
			ciDescriptor.setPermissionSchemeValueInConfigurationItem(copyOfCi, psValue);
		}
	}

	/**
	 * Converts the string representation of the value to an object of the appropriate type for the given property, if
	 * possible.
	 */
	public static Object toPropertyValue(ConfigurationItemPropertyDescriptor descriptor, String value) {
		switch (descriptor.getType()) {
		case BOOLEAN:
			// don't want to use Boolean.valueOf as that simply returns false
			// for invalid values
			return BooleanUtils.toBoolean(value, "true", "false");
		case STRING:
			return value;
		case INTEGER:
			return toInteger(value);
		case ENUM:
			return toEnumValue(descriptor.getPropertyClass(), value);
		default:
			throw new IllegalArgumentException(String.format("Unsupported type: %s for property %s.%s", descriptor.getType(), descriptor.getPropertyField()
					.getDeclaringClass().getName(), descriptor.getPropertyField().getName()));
		}
	}

	private static Integer toInteger(String value) {
		return ((value == null) ? 0 : Integer.valueOf(value));
	}

	@SuppressWarnings("unchecked")
	private static Enum<?> toEnumValue(Class<?> enumClass, String value) {
		return ((value == null) ? null : Enum.valueOf((Class<Enum>) enumClass, value));
	}

}
