package com.xebialabs.deployit.hostsession;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.commons.lang.StringUtils;

/**
 * An implementation of the {@link CommandExecutionCallbackHandler} that captures the output in variables.
 */
public final class CapturingCommandExecutionCallbackHandler implements CommandExecutionCallbackHandler {

	private List<String> outputLines = new ArrayList<String>();

	private List<String> errorLines = new ArrayList<String>();

	private List<String> allLines = new ArrayList<String>();

	public void handleOutputLine(String line) {
		outputLines.add(line);
		allLines.add(line);
	}

	public void handleErrorLine(String line) {
		errorLines.add(line);
		allLines.add(line);
	}

	public void handleOutput(char c) {
		// no-op
	}

	/**
	 * Returns the captured regular output lines.
	 * 
	 * @return a list of captured regular output lines.
	 */
	public List<String> getOutputLines() {
		return Collections.unmodifiableList(outputLines);
	}

	/**
	 * Returns the captured regular output, with the lines separated by "\n".
	 * 
	 * @return the captured regular output as one string.
	 */
	public String getOutput() {
		return StringUtils.join(outputLines, "\n");
	}

	/**
	 * Returns the captured error output lines.
	 * 
	 * @return a list of captured error output lines.
	 */
	public List<String> getErrorLines() {
		return Collections.unmodifiableList(errorLines);
	}

	/**
	 * Returns the captured error output, with the lines separated by "\n".
	 * 
	 * @return the captured error output as one string.
	 */
	public String getError() {
		return StringUtils.join(errorLines, "\n");
	}

	/**
	 * Returns the captured regular and error output lines.
	 * 
	 * @return a list of captured error regular and output lines.
	 */
	public List<String> getAllLines() {
		return Collections.unmodifiableList(allLines);
	}

	/**
	 * Returns the captured regular and error output, with the lines separated by "\n".
	 * 
	 * @return the captured regular and error output as one string.
	 */
	public String getAll() {
		return StringUtils.join(allLines, "\n");
	}

}
