package com.xebialabs.deployit.hostsession;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

import com.xebialabs.deployit.exception.RuntimeIOException;

/**
 * Contains a number of convenience methods for working with HostFile objects.
 */
public class HostFileUtils {

	/**
	 * Copies a file or directory.
	 * 
	 * @param src
	 *            the source file or directory
	 * @param dst
	 *            the destination file or directory
	 */
	public static void copy(HostFile src, HostFile dst) {
		if (src.isDirectory()) {
			copyDirectory(src, dst);
		} else {
			copyFile(src, dst);
		}

	}

	/**
	 * Copies a regular file.
	 * 
	 * @param srcFile
	 *            the source file
	 * @param dstFile
	 *            the destination file
	 */
	public static void copyFile(HostFile srcFile, HostFile dstFile) throws RuntimeIOException {
		HostFile destinationFile;
		if (dstFile.exists() && dstFile.isDirectory())
			destinationFile = dstFile.getSession().getFile(dstFile, srcFile.getName());
		else
			destinationFile = dstFile;

		InputStream in = srcFile.get();
		try {
			destinationFile.put(in, srcFile.length());
		} finally {
			IOUtils.closeQuietly(in);
		}
	}

	/**
	 * Copies a directory recursively.
	 * 
	 * @param srcDir
	 *            the source directory
	 * @param dstDir
	 *            the destination directory
	 */
	public static void copyDirectory(HostFile srcDir, HostFile dstDir) throws RuntimeIOException {
		// check source directory
		if (!srcDir.exists()) {
			throw new RuntimeIOException("Source directory " + srcDir + " does not exist");
		}
		if (!srcDir.isDirectory()) {
			throw new RuntimeIOException("Source directory " + srcDir + " exists but is not a directory");
		}

		// check destination directory, and create it if it is not there yet
		if (dstDir.exists()) {
			if (!dstDir.isDirectory()) {
				throw new RuntimeIOException("Target directory " + dstDir + " already exists and is not a directory");
			}
		} else {
			dstDir.mkdirs();
		}

		List<HostFile> srcFiles = srcDir.listFiles();
		for (HostFile srcFile : srcFiles) {
			HostFile dstFile = dstDir.getFile(srcFile.getName());
			copy(srcFile, dstFile);
		}
	}

	/**
	 * Copies the contents of a string to a {@link HostFile}.
	 * 
	 * @param sourceString
	 *            the string to copy.
	 * @param targetFile
	 *            the host file to copy to.
	 */
	public static void putStringToHostFile(String sourceString, HostFile targetFile) {
		byte[] bytes;
		try {
			bytes = sourceString.getBytes("UTF-8");
		} catch (UnsupportedEncodingException exc) {
			throw new RuntimeIOException(exc);
		}

		InputStream in = new ByteArrayInputStream(bytes);
		try {
			targetFile.put(in, bytes.length);
		} finally {
			IOUtils.closeQuietly(in);
		}
	}

	/**
	 * Reads the contents of a {@link HostFile} as a string.
	 * 
	 * @param sourceHostFile
	 *            the host file to read.
	 * @return the contents of the host file.
	 */
	public static String getHostFileAsString(HostFile sourceHostFile) {
		String content;
		OutputStream os = new ByteArrayOutputStream();
		try {
			sourceHostFile.get(os);
			String unformattedContent = os.toString();
			content = new String(unformattedContent.getBytes("UTF-8"));
		} catch (UnsupportedEncodingException exc) {
			throw new RuntimeIOException(exc);
		} finally {
			IOUtils.closeQuietly(os);
		}
		return content;
	}

	/**
	 * Unzips a host file.
	 * 
	 * @param zip
	 *            the file to unzip.
	 * @param dir
	 *            the directory to unzip to.
	 */
	public static void unzip(HostFile zip, HostFile dir) {
		if (!dir.isDirectory()) {
			throw new IllegalArgumentException(dir + " is not a directory");
		}
		unzip(zip.getPath(), zip.get(), dir);
	}

	private static void unzip(String zipPath, InputStream zipStream, HostFile dir) {
		if (logger.isDebugEnabled())
			logger.debug("Unzipping " + zipPath);

		ZipInputStream zipIn = new ZipInputStream(zipStream);
		try {
			try {
				ZipEntry entry;
				while ((entry = zipIn.getNextEntry()) != null) {
					uploadZipEntry(zipIn, entry, dir);
				}
			} catch (IOException exc) {
				throw new RuntimeIOException("Cannot unzip " + zipPath + " to " + dir, exc);
			}
		} finally {
			IOUtils.closeQuietly(zipIn);
		}
	}

	private static void uploadZipEntry(ZipInputStream zipIn, ZipEntry entry, HostFile dir) {
		String[] pathElements = StringUtils.split(entry.getName(), "/");
		String dstFileSeparator = dir.getSession().getHostOperatingSystem().getFileSeparator();
		String dstPath = StringUtils.join(pathElements, dstFileSeparator);
		HostFile dstFile = dir.getFile(dstPath);

		if (logger.isDebugEnabled())
			logger.debug("Unzipping " + entry.getName() + " to " + dstFile.getPath());

		if (entry.isDirectory()) {
			dstFile.mkdirs();
		} else {
			dstFile.getParentFile().mkdirs();
			dstFile.put(zipIn, entry.getSize());
		}
	}

	private static Logger logger = Logger.getLogger(HostFileUtils.class);
}
