package com.xebialabs.deployit.hostsession;

import com.xebialabs.deployit.ci.Host;
import com.xebialabs.deployit.ci.HostAccessMethod;
import com.xebialabs.deployit.ci.OperatingSystemFamily;
import com.xebialabs.deployit.hostsession.local.LocalHostSession;
import com.xebialabs.deployit.hostsession.ssh.SshScpHostSession;
import com.xebialabs.deployit.hostsession.ssh.SshSftpHostSession;
import com.xebialabs.deployit.hostsession.ssh.SshSudoHostSession;

/**
 * Factory for {@link HostSession host sessions}.
 */
public class HostSessionFactory {

	/**
	 * The default path of the directory in which to store temporary files
	 */
	public static final String DEFAULT_TEMPORARY_DIRECTORY_PATH = "/tmp";

	/**
	 * The default port to use for SSH host sessions
	 */
	public static final int DEFAULT_SSH_PORT = 22;

	/**
	 * Creates a host session for the host.
	 * 
	 * @param osFamily
	 *            the OS family of the host
	 * @param accessMethod
	 *            the way the host is accessed, e.g. local access, SSH w/ SFTP, SSH w/ SCP, SSH w/ SUDO, etc.
	 * @param address
	 *            the address of the host
	 * @param port
	 *            the port on the host to connect with
	 * @param username
	 *            the username to connect to the host
	 * @param password
	 *            the password to connect to the host
	 * @param sudoUsername
	 *            the username to sudo to
	 * @param temporaryDirectoryPath
	 *            the path of the directory in which to store temporary files
	 * @return the session created
	 * @throws IllegalStateException
	 *             if no suitable session can be created.
	 */
	public static HostSession getHostSession(OperatingSystemFamily osFamily, HostAccessMethod accessMethod, String address, int port, String username,
			String password, String sudoUsername, String temporaryDirectoryPath) {
		HostSession s;
		switch (accessMethod) {
		case LOCAL:
			s = new LocalHostSession(osFamily, temporaryDirectoryPath);
			break;
		case SSH_SFTP:
			s = new SshSftpHostSession(osFamily, temporaryDirectoryPath, address, port, username, password);
			break;
		case SSH_SCP:
			s = new SshScpHostSession(osFamily, temporaryDirectoryPath, address, port, username, password);
			break;
		case SSH_SUDO:
			s = new SshSudoHostSession(osFamily, temporaryDirectoryPath, address, port, username, password, sudoUsername);
			break;
		default:
			throw new IllegalStateException("Unknown host access method " + accessMethod);
		}
		return s;
	}

	/**
	 * Creates a host session for the host.
	 * 
	 * @param osFamily
	 *            the OS family of the host
	 * @param accessMethod
	 *            the way the host is accessed, e.g. local access, SSH w/ SFTP, SSH w/ SCP, SSH w/ SUDO, etc.
	 * @param hostSpecification
	 *            the host to connect with, specified as a host address optionally followed by a colon and a port number
	 * @param username
	 *            the username to connect to the host
	 * @param password
	 *            the password to connect to the host
	 * @param sudoUsername
	 *            the username to sudo to
	 * @param temporaryDirectoryPath
	 *            the path of the directory in which to store temporary files
	 * @return the session created
	 * @throws IllegalArgumentException
	 *             if the host specification contains an error
	 * @throws IllegalStateException
	 *             if no suitable session can be created.
	 */
	public static HostSession getHostSession(OperatingSystemFamily osFamily, HostAccessMethod accessMethod, String hostSpecification, String username,
			String password, String sudoUsername, String temporaryDirectoryPath) {
		// get address and port
		String address;
		int port;
		int pos = hostSpecification.indexOf(':');
		if (pos > 0 && pos < hostSpecification.length()) {
			address = hostSpecification.substring(0, pos);
			try {
				port = Integer.parseInt(hostSpecification.substring(pos + 1));
			} catch (NumberFormatException ignore) {
				throw new IllegalArgumentException("Host specification " + hostSpecification + " has an invalid port number");
			}
		} else {
			address = hostSpecification;
			port = DEFAULT_SSH_PORT;
		}

		if (temporaryDirectoryPath == null || temporaryDirectoryPath.length() == 0) {
			temporaryDirectoryPath = getDefaultTemporaryDirectoryPath(accessMethod);
		}

		return getHostSession(osFamily, accessMethod, address, port, username, password, sudoUsername, temporaryDirectoryPath);
	}

	/**
	 * Returns an HostSession based on the information in a Host CI.
	 * 
	 * @param host
	 *            the host from which to take the details.
	 * @return the session created
	 * @throws IllegalArgumentException
	 *             if the host specification contains an error
	 * @throws IllegalStateException
	 *             if no suitable session can be created.
	 */
	public static HostSession getHostSession(Host host) {
		return getHostSession(host.getOperatingSystemFamily(), host.getAccessMethod(), host.getAddress(), host.getUsername(), host.getPassword(), host
				.getSudoUsername(), host.getTemporaryDirectoryLocation());
	}

	private static String getDefaultTemporaryDirectoryPath(HostAccessMethod accessMethod) {
		if (accessMethod == HostAccessMethod.LOCAL) {
			return System.getProperty("java.io.tmpdir", DEFAULT_TEMPORARY_DIRECTORY_PATH);
		} else {
			return DEFAULT_TEMPORARY_DIRECTORY_PATH;
		}
	}

}
