package com.xebialabs.deployit.hostsession.common;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.io.IOUtils;
import org.apache.log4j.Logger;
import org.springframework.core.io.Resource;

import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.hostsession.HostFile;
import com.xebialabs.deployit.hostsession.HostSession;

/**
 * Abstract base class with common methods used by actual implementations of {@link HostFile}.
 */
public abstract class AbstractHostFile implements HostFile {

	protected HostSession session;

	public AbstractHostFile(HostSession session) {
		this.session = session;
	}

	public HostSession getSession() {
		return session;
	}

	public HostFile getFile(String name) {
		return getSession().getFile(this, name);
	}

	public HostFile getParentFile() {
		String parent = getParent();
		if (parent == null || parent.length() == 0) {
			return null;
		} else {
			return getSession().getFile(parent);
		}
	}

	public List<HostFile> listFiles() throws RuntimeIOException {
		List<String> filenames = list();
		if (filenames == null) {
			return null;
		} else {
			List<HostFile> listFiles = new ArrayList<HostFile>(filenames.size());
			for (String filename : filenames) {
				listFiles.add(getSession().getFile(this, filename));
			}
			return listFiles;
		}
	}

	public boolean deleteRecursively() throws RuntimeIOException {
		if (!exists()) {
			return false;
		} else {
			deleteRecursivelyWithoutExistenceCheck(this);
			return true;
		}
	}

	private static void deleteRecursivelyWithoutExistenceCheck(HostFile d) throws RuntimeIOException {
		if (d.isDirectory()) {
			List<HostFile> contents = d.listFiles();
			for (HostFile f : contents) {
				AbstractHostFile.deleteRecursivelyWithoutExistenceCheck(f);
			}
		}

		// The SSH implementation of delete actually redoes the existence check :-(
		d.delete();
	}

	public boolean deleteRecursively(String exclusion) throws RuntimeIOException {
		if (!exists()) {
			return false;
		} else {
			deleteRecursivelyWithoutExistenceCheck(this, exclusion);
			return true;
		}
	}

	private static void deleteRecursivelyWithoutExistenceCheck(HostFile d, String exclusion) throws RuntimeIOException {
		if (excludeFilter(d, exclusion)) {
			return;
		}
		if (d.isDirectory()) {
			List<HostFile> contents = d.listFiles();
			for (HostFile f : contents) {
				AbstractHostFile.deleteRecursivelyWithoutExistenceCheck(f, exclusion);
			}
		}

		if (d.isDirectory()) {
			// must be empty otherwise we can't delete
			List<HostFile> contents = d.listFiles();
			if (contents == null || contents.size() == 0) {
				d.delete();
			}
		} else {
			d.delete();
		}
	}

	private static boolean excludeFilter(HostFile d, String exclusion) {
		if (exclusion == null) {
			return false;
		}
		String[] pieces = exclusion.split(",");
		for (String piece : pieces) {
			if (d.getName().equals(piece.trim())) {
				return true;
			}
		}
		return false;
	}

	public void get(OutputStream out) throws RuntimeIOException {
		try {
			InputStream in = get();
			try {
				IOUtils.copy(in, out);
			} finally {
				IOUtils.closeQuietly(in);
			}
		} catch (IOException exc) {
			throw new RuntimeIOException(exc);
		}
	}

	public void get(File file) throws RuntimeIOException {
		try {
			OutputStream out = new FileOutputStream(file);
			try {
				get(out);
			} finally {
				out.close();
			}
		} catch (IOException exc) {
			throw new RuntimeIOException(exc);
		}
	}

	public void put(InputStream in, long length) throws RuntimeIOException {
		try {
			OutputStream out = put(length);
			try {
				IOUtils.copy(in, out);
			} finally {
				IOUtils.closeQuietly(out);
			}
		} catch (IOException exc) {
			throw new RuntimeIOException(exc);
		}
	}

	public void put(File file) throws RuntimeIOException {
		try {
			InputStream in = new FileInputStream(file);
			try {
				put(in, file.length());
			} finally {
				in.close();
			}
		} catch (IOException exc) {
			throw new RuntimeIOException(exc);
		}
	}

	public void put(Resource resource) throws RuntimeIOException {
		try {
			byte[] resourceBytes = IOUtils.toByteArray(resource.getInputStream());
			OutputStream out = put(resourceBytes.length);
			try {
				IOUtils.write(resourceBytes, out);
			} finally {
				out.close();
			}
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	@SuppressWarnings("unused")
	private static Logger logger = Logger.getLogger(AbstractHostFile.class);

}
