package com.xebialabs.deployit.hostsession.local;

import java.io.File;
import java.io.IOException;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

import com.xebialabs.deployit.ci.OperatingSystemFamily;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.hostsession.CommandExecutionCallbackHandler;
import com.xebialabs.deployit.hostsession.HostFile;
import com.xebialabs.deployit.hostsession.HostSession;
import com.xebialabs.deployit.hostsession.common.AbstractHostSession;
import com.xebialabs.deployit.hostsession.common.ErrorStreamToCallbackHandler;
import com.xebialabs.deployit.hostsession.common.InputResponseHandler;
import com.xebialabs.deployit.hostsession.common.OutputStreamToCallbackHandler;

/**
 * A session to the local host.
 */
public class LocalHostSession extends AbstractHostSession implements HostSession {

	/**
	 * Constructs a session to the local host.
	 */
	public LocalHostSession(OperatingSystemFamily os, String temporaryDirectoryPath) {
		super(os, temporaryDirectoryPath);
	}

	public HostFile getTempFile(String prefix, String suffix) throws RuntimeIOException {
		try {
			return new LocalHostFile(this, File.createTempFile(prefix, suffix, new File(getTemporaryDirectory().getPath())));
		} catch (IOException exc) {
			throw new RuntimeIOException(exc);
		}
	}

	public HostFile getFile(String hostPath) throws RuntimeIOException {
		return new LocalHostFile(this, new File(hostPath));
	}

	public HostFile getFile(HostFile parent, String child) throws RuntimeIOException {
		if (!(parent instanceof LocalHostFile)) {
			throw new IllegalStateException("parent is not a file on the local host");
		}
		File parentFile = ((LocalHostFile) parent).getFile();
		return new LocalHostFile(this, new File(parentFile, child));
	}

	public int execute(CommandExecutionCallbackHandler handler, String... commandLine) throws RuntimeIOException {
		Process proc;
		try {
			logger.info("Executing local command \"" + StringUtils.join(commandLine, " ") + "\"");
			proc = Runtime.getRuntime().exec(commandLine);
		} catch (IOException exc) {
			throw new RuntimeIOException(exc);
		}

		Thread outputGobblerThread = new Thread(new OutputStreamToCallbackHandler(proc.getInputStream(), handler));
		outputGobblerThread.start();

		Thread errorGobblerThread = new Thread(new ErrorStreamToCallbackHandler(proc.getErrorStream(), handler));
		errorGobblerThread.start();

		try {
			return proc.waitFor();
		} catch (InterruptedException exc) {
			throw new RuntimeIOException("Execution was interrupted waiting for the command to complete", exc);
		}
	}

	public int execute(CommandExecutionCallbackHandler handler, Map<String, String> inputResponse, String... commandLine) {
		Process proc;
		try {
			logger.info("Executing local command \"" + StringUtils.join(commandLine, " ") + "\"");
			proc = Runtime.getRuntime().exec(commandLine);
		} catch (IOException exc) {
			throw new RuntimeIOException(exc);
		}

		InputResponseHandler responseHandler = new InputResponseHandler(handler, proc.getOutputStream(), inputResponse);
		Thread outputGobblerThread = new Thread(new OutputStreamToCallbackHandler(proc.getInputStream(), responseHandler));
		outputGobblerThread.start();

		Thread errorGobblerThread = new Thread(new ErrorStreamToCallbackHandler(proc.getErrorStream(), responseHandler));
		errorGobblerThread.start();

		try {
			return proc.waitFor();
		} catch (InterruptedException exc) {
			throw new RuntimeIOException("Execution was interrupted waiting for the command to complete", exc);
		}
	}

	/**
	 * Returns a string representation of the local host session in scp syntax: <code>localhost:</code>
	 */
	public String toString() {
		return "localhost";
	}

	private Logger logger = Logger.getLogger(LocalHostSession.class);

}
