package com.xebialabs.deployit.hostsession.ssh;

import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;
import java.util.StringTokenizer;

import org.apache.log4j.Logger;

import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.hostsession.CapturingCommandExecutionCallbackHandler;
import com.xebialabs.deployit.hostsession.CommandExecutionCallbackHandler;
import com.xebialabs.deployit.hostsession.HostFile;

/**
 * A file on a host connected through SSH w/ SCP.
 */
class SshScpHostFile extends SshHostFile implements HostFile {

	/**
	 * Constructs a SshScpHostFile
	 * 
	 * @param session
	 *            the session connected to the host
	 * @param remotePath
	 *            the path of the file on the host
	 */
	public SshScpHostFile(SshHostSession session, String remotePath) {
		super(session, remotePath);
	}

	public boolean exists() throws RuntimeIOException {
		return stat().exists;
	}

	public boolean isDirectory() throws RuntimeIOException {
		return stat().isDirectory;
	}

	public long length() throws RuntimeIOException {
		return stat().length;
	}

	public boolean canRead() throws RuntimeIOException {
		return stat().canRead;
	}

	public boolean canWrite() throws RuntimeIOException {
		return stat().canWrite;
	}

	public boolean canExecute() throws RuntimeIOException {
		return stat().canExecute;
	}

	/**
	 * Holds results of an ls call
	 */
	private static class StatResults {
		public boolean exists;
		public boolean isDirectory;
		public long length = -1;
		public boolean canRead;
		public boolean canWrite;
		public boolean canExecute;
	}

	protected StatResults stat() throws RuntimeIOException {
		StatResults results = new StatResults();
		CapturingCommandExecutionCallbackHandler capturedOutput = new CapturingCommandExecutionCallbackHandler();
		int errno = executeCommand(capturedOutput, "ls", "-ld", remotePath);
		if (errno == 0) {
			results.exists = true;
			if (capturedOutput.getOutputLines().size() > 0) {
				// parse ls results
				String outputLine = capturedOutput.getOutputLines().get(0);
				if (logger.isDebugEnabled())
					logger.debug("ls output = " + outputLine);
				StringTokenizer outputTokens = new StringTokenizer(outputLine);
				if (outputTokens.countTokens() < 5) {
					throw new RuntimeIOException("ls -ld " + remotePath + " returned output that contains less than the expected 5 tokens");
				}
				String permissions = outputTokens.nextToken();
				@SuppressWarnings("unused")
				String inodelinkes = outputTokens.nextToken();
				@SuppressWarnings("unused")
				String owner = outputTokens.nextToken();
				@SuppressWarnings("unused")
				String group = outputTokens.nextToken();
				String size = outputTokens.nextToken();

				results.isDirectory = permissions.length() >= 1 && permissions.charAt(0) == 'd';
				try {
					results.length = Integer.parseInt(size);
				} catch (NumberFormatException ignore) {
				}
			}
		} else {
			results.exists = false;
		}

		if (logger.isDebugEnabled())
			logger.debug("Statted remote file " + this + " (exists=" + results.exists + ", isDirectory=" + results.isDirectory + ", length=" + results.length + ")");
		return results;
	}

	public List<String> list() throws RuntimeIOException {
		CapturingCommandExecutionCallbackHandler capturedOutput = new CapturingCommandExecutionCallbackHandler();
		// yes, this *is* meant to be 'el es min one'!
		int errno = executeCommand(capturedOutput, "ls", "-1", remotePath);
		if (errno != 0) {
			throw new RuntimeIOException("Cannot list remote directory " + this + ": " + capturedOutput.getError() + " (errno=" + errno + ")");
		}

		if (logger.isDebugEnabled())
			logger.debug("Listed remote directory " + this);
		return capturedOutput.getOutputLines();
	}

	public void mkdir() throws RuntimeIOException {
		CapturingCommandExecutionCallbackHandler capturedOutput = new CapturingCommandExecutionCallbackHandler();
		int errno = executeCommand(capturedOutput, "mkdir", remotePath);
		if (errno != 0) {
			throw new RuntimeIOException("Cannot create remote directory " + this + ": " + capturedOutput.getError() + " (errno=" + errno + ")");
		}
		if (logger.isDebugEnabled())
			logger.debug("Created remote directory " + this);
	}

	public void mkdirs() throws RuntimeIOException {
		CapturingCommandExecutionCallbackHandler capturedOutput = new CapturingCommandExecutionCallbackHandler();
		int errno = executeCommand(capturedOutput, "mkdir", "-p", remotePath);
		if (errno != 0) {
			throw new RuntimeIOException("Cannot create remote directory (or parent directories) " + this + ": " + capturedOutput.getError() + " (errno=" + errno + ")");
		}
		if (logger.isDebugEnabled())
			logger.debug("Created remote directory and parents if necessary" + this);
	}

	@Override
	protected void deleteDirectory() {
		CapturingCommandExecutionCallbackHandler capturedOutput = new CapturingCommandExecutionCallbackHandler();
		int errno = executeCommand(capturedOutput, "rmdir", remotePath);
		if (errno != 0) {
			throw new RuntimeIOException("Cannot delete remote directory " + this + ": " + capturedOutput.getError() + " (errno=" + errno + ")");
		}
		if (logger.isDebugEnabled())
			logger.debug("Deleted remote directory " + this);
	}

	@Override
	protected void deleteFile() {
		CapturingCommandExecutionCallbackHandler capturedOutput = new CapturingCommandExecutionCallbackHandler();
		int errno = executeCommand(capturedOutput, "rm", remotePath);
		if (errno != 0) {
			throw new RuntimeIOException("Cannot delete remote file " + this + ": " + capturedOutput.getError() + " (errno=" + errno + ")");
		}
		if (logger.isDebugEnabled())
			logger.debug("Deleted remote file " + this);
	}

	@Override
	public boolean deleteRecursively() throws RuntimeIOException {
		if (!exists()) {
			return false;
		} else {
			CapturingCommandExecutionCallbackHandler capturedOutput = new CapturingCommandExecutionCallbackHandler();
			int errno = executeCommand(capturedOutput, "rm", "-r", remotePath);
			if (errno != 0) {
				throw new RuntimeIOException("Cannot recursively delete remote directory " + this + ": " + capturedOutput.getError() + " (errno=" + errno + ")");
			}
			if (logger.isDebugEnabled())
				logger.debug("Recursively deleted remote file or directory " + this);
			return true;
		}
	}

	protected int executeCommand(CommandExecutionCallbackHandler handler, String... command) {
		return sshHostSession.execute(handler, command);
	}

	public InputStream get() throws RuntimeIOException {
		SshScpInputStream in = new SshScpInputStream(this);
		in.open();
		if (logger.isDebugEnabled())
			logger.debug("Opened SCP input stream from remote file " + this);
		return in;
	}

	public OutputStream put(long length) throws RuntimeIOException {
		SshScpOutputStream out = new SshScpOutputStream(this, length);
		out.open();
		if (logger.isDebugEnabled())
			logger.debug("Opened SCP output stream to remote file " + this);
		return out;
	}

	private Logger logger = Logger.getLogger(SshScpHostFile.class);

}
