package com.xebialabs.deployit.hostsession.ssh;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import org.apache.commons.io.IOUtils;
import org.apache.log4j.Logger;

import com.jcraft.jsch.ChannelExec;
import com.jcraft.jsch.JSchException;
import com.jcraft.jsch.Session;
import com.xebialabs.deployit.exception.RuntimeIOException;

/**
 * An output stream to a file on a host connected through SSH w/ SCP.
 */
class SshScpOutputStream extends OutputStream {

	protected SshScpHostFile file;

	protected long length;

	protected String command;

	protected Session session;

	protected ChannelExec channel;

	protected InputStream channelIn;

	protected OutputStream channelOut;


	private static final String CHANNEL_PURPOSE = " (for SCP input stream)";

	SshScpOutputStream(SshScpHostFile file, long length) {
		this.file = file;
		this.length = length;
	}

	void open() {
		try {
			// connect to SSH and start scp in sink mode
			session = file.sshHostSession.openSession(CHANNEL_PURPOSE);
			channel = (ChannelExec) session.openChannel("exec");
			// no password in this command, so use 'false'
			command = SshHostSession.encodeCommandLine(false, "scp", "-t", file.remotePath);
			channel.setCommand(command);
			channelIn = channel.getInputStream();
			channelOut = channel.getOutputStream();
			channel.connect();
			logger.info("Executing remote command \"" + command + "\" on " + file.sshHostSession + " to open SCP stream for writing");

			// perform SCP write protocol
			readAck();
			sendFilePreamble();
			logger.info("Opened SCP stream to write to remote file " + file);
		} catch (IOException exc) {
			throw new RuntimeIOException("Cannot open SCP stream to write remote file " + file, exc);
		} catch (JSchException exc) {
			throw new RuntimeIOException("Cannot open SCP stream to write remote file " + file, exc);
		}
	}

	private void readAck() {
		if (logger.isDebugEnabled())
			logger.debug("Reading ACK");

		int c = SshStreamUtils.checkAck(channelIn);
		if (c != 0) {
			throw new RuntimeIOException("Protocol error on SCP stream to write remote file " + file);
		}
	}

	private void sendFilePreamble() throws IOException {
		String preamble = "C0644 " + length + " ";
		if (file.remotePath.lastIndexOf('/') > 0) {
			preamble += file.remotePath.substring(file.remotePath.lastIndexOf('/') + 1);
		} else {
			preamble += file.remotePath;
		}
		if (logger.isDebugEnabled())
			logger.debug("Sending file preamble \"" + preamble + "\"");
		preamble += "\n";

		channelOut.write(preamble.getBytes());
		channelOut.flush();
	}

	private void sendAck() throws IOException {
		if (logger.isDebugEnabled())
			logger.debug("Sending ACK");

		channelOut.write('\0');
		channelOut.flush();
	}

	@Override
	public void write(byte[] b) throws IOException {
		channelOut.write(b);
	}

	@Override
	public void write(byte[] b, int off, int len) throws IOException {
		channelOut.write(b, off, len);
	}

	@Override
	public void write(int b) throws IOException {
		channelOut.write(b);
	}

	@Override
	public void close() {
		try {
			sendAck();
			readAck();
		} catch (IOException ignore) {
		}

		// close output channel to force remote scp to quit
		IOUtils.closeQuietly(channelOut);

		// get return code from remote scp
		int res = SshHostSession.waitForExitStatus(channel);

		IOUtils.closeQuietly(channelIn);
		channel.disconnect();
		file.sshHostSession.disconnectSession(session, CHANNEL_PURPOSE);

		if (res != 0) {
			throw new RuntimeIOException("Error closing SCP stream to write remote file " + file + " (remote scp command returned error code " + res + ")");
		}
	}

	private Logger logger = Logger.getLogger(SshScpOutputStream.class);

}
