package com.xebialabs.deployit.hostsession.ssh;

import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import org.apache.log4j.Logger;

import com.jcraft.jsch.ChannelSftp;
import com.jcraft.jsch.JSchException;
import com.jcraft.jsch.SftpATTRS;
import com.jcraft.jsch.SftpException;
import com.jcraft.jsch.ChannelSftp.LsEntry;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.hostsession.HostFile;

/**
 * A file on a host connected through SSH that is accessed using SFTP.
 */
class SshSftpHostFile extends SshHostFile {

	private SshSftpHostSession sshSftpHostSession;

	public SshSftpHostFile(SshSftpHostSession session, String remotePath) {
		super(session, remotePath);
		sshSftpHostSession = session;
	}

	protected SftpATTRS stat() throws RuntimeIOException {
		try {
			SftpATTRS attrs = sshSftpHostSession.getSharedSftpChannel().stat(remotePath);
			if (logger.isDebugEnabled())
				logger.debug("Statted remote file " + this);
			return attrs;
		} catch (SftpException exc) {
			throw new RuntimeIOException("Cannot stat remote file " + this, exc);
		} catch (JSchException exc) {
			throw new RuntimeIOException("Cannot stat remote file " + this, exc);
		}
	}

	public boolean exists() throws RuntimeIOException {
		return exists(remotePath);
	}

	private boolean exists(String path) throws RuntimeIOException {
		try {
			sshSftpHostSession.getSharedSftpChannel().stat(path);
			if (logger.isDebugEnabled())
				logger.debug("Checked remote file " + path + " for existence and found it");
			return true;
		} catch (SftpException exc) {
			// if we get an SftpException while trying to stat the file, we
			// assume it does not exist
			if (logger.isDebugEnabled())
				logger.debug("Checked remote file " + path + " for existence and did not find it");
			return false;
		} catch (JSchException exc) {
			throw new RuntimeIOException("Cannot check existence of remote file " + path, exc);
		}
	}

	public boolean isDirectory() throws RuntimeIOException {
		return stat().isDir();
	}

	public long length() throws RuntimeIOException {
		return stat().getSize();
	}

	public boolean canExecute() throws RuntimeIOException {
		SftpATTRS attrs = stat();
		return (attrs.getPermissions() | 0100) != 0;
	}

	public boolean canRead() throws RuntimeIOException {
		SftpATTRS attrs = stat();
		return (attrs.getPermissions() | 0400) != 0;
	}

	public boolean canWrite() throws RuntimeIOException {
		SftpATTRS attrs = stat();
		return (attrs.getPermissions() | 0200) != 0;
	}

	public List<String> list() throws RuntimeIOException {
		try {
			// read files from host
			@SuppressWarnings("unchecked")
			Vector<LsEntry> ls = (Vector<LsEntry>) sshSftpHostSession.getSharedSftpChannel().ls(remotePath);

			// copy files to list, skipping . and ..
			List<String> filenames = new ArrayList<String>(ls.size());
			for (LsEntry lsEntry : ls) {
				String filename = lsEntry.getFilename();
				if (filename.equals(".") || filename.equals("..")) {
					continue;
				}
				filenames.add(filename);
			}
			return filenames;
		} catch (SftpException exc) {
			throw new RuntimeIOException("Cannot list remote directory " + this + ": " + exc.toString(), exc);
		} catch (JSchException exc) {
			throw new RuntimeIOException("Cannot list remote directory " + this + ": " + exc.toString(), exc);
		}
	}

	public void mkdir() throws RuntimeIOException {
		mkdir(remotePath);
	}

	private void mkdir(String path) throws RuntimeIOException {
		try {
			sshSftpHostSession.getSharedSftpChannel().mkdir(path);
			if (logger.isDebugEnabled())
				logger.debug("Created remote directory " + path);
		} catch (SftpException exc) {
			throw new RuntimeIOException("Cannot create remote directory " + path + ": " + exc.toString(), exc);
		} catch (JSchException exc) {
			throw new RuntimeIOException("Cannot create remote directory " + path + ": " + exc.toString(), exc);
		}
	}

	public void mkdirs() throws RuntimeIOException {
		List<HostFile> allDirs = new ArrayList<HostFile>();
		HostFile dir = this;
		do {
			allDirs.add(0, dir);
		} while ((dir = dir.getParentFile()) != null);

		for (HostFile each : allDirs) {
			if (!each.exists()) {
				each.mkdir();
			}
		}
	}

	@Override
	protected void deleteFile() {
		try {
			sshSftpHostSession.getSharedSftpChannel().rm(remotePath);
			if (logger.isDebugEnabled())
				logger.debug("Removed remote file " + this);
		} catch (SftpException exc) {
			throw new RuntimeIOException("Cannot delete remote file " + this + ": " + exc.toString(), exc);
		} catch (JSchException exc) {
			throw new RuntimeIOException("Cannot delete remote file " + this + ": " + exc.toString(), exc);
		}
	}

	@Override
	protected void deleteDirectory() {
		try {
			sshSftpHostSession.getSharedSftpChannel().rmdir(remotePath);
			if (logger.isDebugEnabled())
				logger.debug("Removed remote directory " + this);
		} catch (SftpException exc) {
			throw new RuntimeIOException("Cannot delete remote directory " + this + ": " + exc.toString(), exc);
		} catch (JSchException exc) {
			throw new RuntimeIOException("Cannot delete remote directory " + this + ": " + exc.toString(), exc);
		}
	}

	public InputStream get() throws RuntimeIOException {
		try {
			ChannelSftp sftpChannel = sshSftpHostSession.openSftpChannel();
			InputStream in = new SshSftpInputStream(sshSftpHostSession, sftpChannel, sftpChannel.get(remotePath));
			if (logger.isDebugEnabled())
				logger.debug("Opened SFTP input stream to read from remote file " + this);
			return in;
		} catch (SftpException exc) {
			throw new RuntimeIOException("Cannot read from remote file " + remotePath + ": " + exc.toString(), exc);
		} catch (JSchException exc) {
			throw new RuntimeIOException("Cannot read from remote file " + remotePath + ": " + exc.toString(), exc);
		}
	}

	public void get(OutputStream out) throws RuntimeIOException {
		try {
			sshSftpHostSession.getSharedSftpChannel().get(remotePath, out);
			if (logger.isDebugEnabled())
				logger.debug("Wrote output stream from remote file " + this);
		} catch (SftpException exc) {
			throw new RuntimeIOException("Cannot read from remote file " + remotePath + ": " + exc.toString(), exc);
		} catch (JSchException exc) {
			throw new RuntimeIOException("Cannot read from remote file " + remotePath + ": " + exc.toString(), exc);
		}
	}

	public OutputStream put(long length) throws RuntimeIOException {
		try {
			ChannelSftp sftpChannel = sshSftpHostSession.openSftpChannel();
			OutputStream out = new SshSftpOutputStream(sshSftpHostSession, sftpChannel, sftpChannel.put(remotePath));
			if (logger.isDebugEnabled())
				logger.debug("Opened SFTP ouput stream to write to remote file " + this);
			return out;
		} catch (SftpException exc) {
			throw new RuntimeIOException("Cannot write to remote file " + remotePath + ": " + exc.toString(), exc);
		} catch (JSchException exc) {
			throw new RuntimeIOException("Cannot write to remote file " + remotePath + ": " + exc.toString(), exc);
		}
	}

	public void put(InputStream in, long length) throws RuntimeIOException {
		try {
			sshSftpHostSession.getSharedSftpChannel().put(in, remotePath);
			if (logger.isDebugEnabled())
				logger.debug("Wrote input stream to remote file " + this);
		} catch (SftpException exc) {
			throw new RuntimeIOException("Cannot write to remote file " + remotePath + ": " + exc.toString(), exc);
		} catch (JSchException exc) {
			throw new RuntimeIOException("Cannot write to remote file " + remotePath + ": " + exc.toString(), exc);
		}
	}

	private static Logger logger = Logger.getLogger(SshSftpHostFile.class);

}
