package com.xebialabs.deployit.hostsession.ssh;

import java.io.InputStream;
import java.io.OutputStream;

import org.apache.commons.io.FilenameUtils;
import org.apache.log4j.Logger;

import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.hostsession.CommandExecutionCallbackHandler;
import com.xebialabs.deployit.hostsession.HostFile;

/**
 * A file on a host connected through SSH w/ SUDO.
 */
class SshSudoHostFile extends SshScpHostFile implements HostFile {

	private SshSudoHostSession sshSudoHostSession;

	private boolean isTempFile;

	/**
	 * Constructs a SshSudoHostFile
	 * 
	 * @param session
	 *            the session connected to the host
	 * @param remotePath
	 *            the path of the file on the host
	 * @param isTempFile
	 *            is <code>true</code> if this is a temporary file; <code>false</code> otherwise
	 */
	public SshSudoHostFile(SshSudoHostSession session, String remotePath, boolean isTempFile) {
		super(session, remotePath);
		this.sshSudoHostSession = session;
		this.isTempFile = isTempFile;
	}

	@Override
	protected int executeCommand(CommandExecutionCallbackHandler handler, String... command) {
		if (isTempFile) {
			return sshSudoHostSession.noSudoExecute(handler, command);
		} else {
			return super.executeCommand(handler, command);
		}
	}

	@Override
	public HostFile getFile(String name) {
		SshSudoHostFile f = (SshSudoHostFile) super.getFile(name);
		f.isTempFile = this.isTempFile;
		return f;
	}

	@Override
	public HostFile getParentFile() {
		SshSudoHostFile f = (SshSudoHostFile) super.getParentFile();
		f.isTempFile = this.isTempFile;
		return f;
	}

	@Override
	public InputStream get() throws RuntimeIOException {
		if (isTempFile) {
			return super.get();
		} else {
			HostFile tempFile = getTempFile(true);
			copyHostFileToTempFile(tempFile);
			return tempFile.get();
		}
	}

	private void copyHostFileToTempFile(HostFile tempFile) {
		if (logger.isDebugEnabled())
			logger.debug("Copying " + this + " to " + tempFile + " for reading");
		session.execute(null, "cp", this.getPath(), tempFile.getPath());
	}

	@Override
	public OutputStream put(long length) throws RuntimeIOException {
		if (isTempFile) {
			return super.put(length);
		} else {
			SshSudoOutputStream out = new SshSudoOutputStream(this, length, getTempFile(false));
			out.open();
			if (logger.isDebugEnabled())
				logger.debug("Opened SUDO output stream to remote file " + this);
			return out;
		}
	}

	protected HostFile getTempFile(boolean useSudoForDeletion) {
		String prefix = FilenameUtils.getBaseName(getPath());
		String suffix = FilenameUtils.getExtension(getPath());
		return sshSudoHostSession.getTempFile(prefix, suffix);
	}

	private Logger logger = Logger.getLogger(SshSudoHostFile.class);

}
