package com.xebialabs.deployit.hostsession.ssh;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;

import com.xebialabs.deployit.ci.OperatingSystemFamily;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.hostsession.CommandExecutionCallbackHandler;
import com.xebialabs.deployit.hostsession.HostFile;

/**
 * A session to a remote host using SSH w/ SUDO.
 */
public class SshSudoHostSession extends SshHostSession {

	protected String sudoUsername;

	/**
	 * Constructs an SshSudoHostSession
	 * 
	 * @param os
	 *            the operating system of the host
	 * @param temporaryDirectoryPath
	 *            the path of the directory in which to store temporary files
	 * @param host
	 *            the hostname or IP adress of the host
	 * @param port
	 *            the port to connect to
	 * @param username
	 *            the username to connect with
	 * @param password
	 *            the password to connect with
	 * @param sudoUsername
	 *            the username to sudo to
	 */
	public SshSudoHostSession(OperatingSystemFamily os, String temporaryDirectoryPath, String host, int port, String username, String password,
			String sudoUsername) {
		super(os, temporaryDirectoryPath, host, port, username, password);
		this.sudoUsername = sudoUsername;
		open();
	}

	@Override
	public int execute(CommandExecutionCallbackHandler handler, Map<String, String> inputResponse, String... commandLine) throws RuntimeIOException {
		// prepend sudo -u <sudoUsername> to command line
		List<String> sudoCommandLine = new ArrayList<String>();
		sudoCommandLine.add("sudo");
		sudoCommandLine.add("-u");
		sudoCommandLine.add(sudoUsername);
		for (String arg : commandLine) {
			sudoCommandLine.add(arg);
		}

		// provide CRLF as initial input to answer any Password prompt
		InputStream input = new ByteArrayInputStream("\r\n".getBytes());

		//
		return super.executeWithInputStream(handler, input, inputResponse, sudoCommandLine.toArray(new String[sudoCommandLine.size()]));
	}

	protected int noSudoExecute(CommandExecutionCallbackHandler handler, String... commandLine) {
		if (logger.isDebugEnabled())
			logger.debug("NOT adding sudo statement");

		return super.execute(handler, commandLine);
	}

	protected HostFile getFile(String hostPath, boolean isTempFile) throws RuntimeIOException {
		return new SshSudoHostFile(this, hostPath, isTempFile);
	}

	public String toString() {
		return username + "@" + host + ":" + port + " (sudo to " + sudoUsername + ")";
	}

	private Logger logger = Logger.getLogger(SshSudoHostSession.class);

}
