package com.xebialabs.deployit.mapper;

import java.io.Serializable;
import java.util.List;

import com.xebialabs.deployit.Change;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.ci.mapping.Mapping;

/**
 * A subclass of {@link StepGeneratingMapper} that provides the same functionality but does not pass the mapping object
 * to the template methods.
 * 
 * @param <S>
 *            the type of the source artifact/resource.
 * @param <T>
 *            the type of the target middleware.
 */
public abstract class MappingAgnosticStepGeneratingMapper<S extends Serializable, T extends Serializable> extends StepGeneratingMapper<S, Mapping, T> {

	/**
	 * Creates a <tt>MappingAgnosticStepGeneratingMapper</tt> for the specified deployment change.
	 * 
	 * @param change
	 *            the deployment change for which to calculate mappings.
	 * @param applyDefaultMappings
	 *            default mappings will be applied iff <code>true</code>
	 */
	public MappingAgnosticStepGeneratingMapper(Change<Deployment> change, boolean applyDefaultMappings) {
		super(change, applyDefaultMappings);
	}

	/**
	 * Creates a <tt>MappingAgnosticStepGeneratingMapper</tt> for the specified deployment change that will apply
	 * default mappings.
	 * 
	 * @param change
	 *            the deployment change for which to calculate mappings.
	 */
	public MappingAgnosticStepGeneratingMapper(Change<Deployment> change) {
		super(change);
	}

	/**
	 * Overrides
	 * {@link StepGeneratingMapper#generateAdditionStepsForAddedMapping(Serializable, Mapping, Serializable, List)} to
	 * invoke {@link #generateAdditionStepsForAddedMapping(Serializable, Serializable, List)}.
	 */
	@Override
	protected final void generateAdditionStepsForAddedMapping(S newMappingSource, Mapping newMapping, T newMappingTarget, List<Step> steps) {
		generateAdditionStepsForAddedMapping(newMappingSource, newMappingTarget, steps);
	}

	/**
	 * Version of
	 * {@link StepGeneratingMapper#generateAdditionStepsForAddedMapping(Serializable, Mapping, Serializable, List)} that
	 * does without the <tt>newMapping</tt> parameter.
	 * 
	 * @param newMappingSource
	 *            the source of the new mapping.
	 * @param newMappingTarget
	 *            the target of the new mapping.
	 * @param steps
	 *            the list of steps to which the generated steps are to be added
	 */
	protected abstract void generateAdditionStepsForAddedMapping(S newMappingSource, T newMappingTarget, List<Step> steps);

	/**
	 * Overrides
	 * {@link StepGeneratingMapper#generateAdditionStepsForModifiedMapping(Serializable, Mapping, Serializable, List)}
	 * to invoke {@link #generateAdditionStepsForModifiedMapping(Serializable, Serializable, List)}.
	 */
	@Override
	protected final void generateAdditionStepsForModifiedMapping(S newMappingSource, Mapping newMapping, T newMappingTarget, List<Step> steps) {
		generateAdditionStepsForModifiedMapping(newMappingSource, newMappingTarget, steps);
	}

	/**
	 * Version of
	 * {@link StepGeneratingMapper#generateAdditionStepsForModifiedMapping(Serializable, Mapping, Serializable, List)}
	 * that does without the <tt>newMapping</tt> parameter. The default implementation invokes
	 * {@link #generateAdditionStepsForAddedMapping(Serializable, Serializable, List)}.
	 * 
	 * @param newMappingSource
	 *            the new source of the modified mapping.
	 * @param newMappingTarget
	 *            the new target of the modified mapping.
	 * @param steps
	 *            the list of steps to which the generated steps are to be added
	 */
	protected void generateAdditionStepsForModifiedMapping(S newMappingSource, T newMappingTarget, List<Step> steps) {
		generateAdditionStepsForAddedMapping(newMappingSource, newMappingTarget, steps);
	}

	/**
	 * Overrides
	 * {@link StepGeneratingMapper#generateDeletionStepsForModifiedMapping(Serializable, Mapping, Serializable, List)}
	 * to invoke {@link #generateDeletionStepsForModifiedMapping(Serializable, Serializable, List)}.
	 */
	@Override
	protected final void generateDeletionStepsForModifiedMapping(S newMappingSource, Mapping newMapping, T newMappingTarget, List<Step> steps) {
		generateDeletionStepsForModifiedMapping(newMappingSource, newMappingTarget, steps);
	}

	/**
	 * Version of
	 * {@link StepGeneratingMapper#generateDeletionStepsForModifiedMapping(Serializable, Mapping, Serializable, List)}
	 * that does without the <tt>oldMapping</tt> parameter. The default implementation invokes
	 * {@link #generateDeletionStepsForDeletedMapping(Serializable, Serializable, List)}.
	 * 
	 * @param oldMappingSource
	 *            the source of the new mapping.
	 * @param oldMappingTarget
	 *            the target of the new mapping.
	 * @param steps
	 *            the list of steps to which the generated steps are to be added
	 */
	protected void generateDeletionStepsForModifiedMapping(S oldMappingSource, T oldMappingTarget, List<Step> steps) {
		generateDeletionStepsForDeletedMapping(oldMappingSource, oldMappingTarget, steps);
	}

	/**
	 * Overrides
	 * {@link StepGeneratingMapper#generateDeletionStepsForDeletedMapping(Serializable, Mapping, Serializable, List)} to
	 * invoke {@link #generateDeletionStepsForDeletedMapping(Serializable, Serializable, List)}.
	 */
	@Override
	protected final void generateDeletionStepsForDeletedMapping(S oldMappingSource, Mapping oldMapping, T oldMappingTarget, List<Step> steps) {
		generateDeletionStepsForDeletedMapping(oldMappingSource, oldMappingTarget, steps);
	}

	/**
	 * Version of
	 * {@link StepGeneratingMapper#generateDeletionStepsForDeletedMapping(Serializable, Mapping, Serializable, List)}
	 * that does without the <tt>oldMapping</tt> parameter.
	 * 
	 * @param oldMappingSource
	 *            the source of the old mapping.
	 * @param oldMappingTarget
	 *            the target of the old mapping.
	 * @param steps
	 *            the list of steps to which the generated steps are to be added
	 */
	protected abstract void generateDeletionStepsForDeletedMapping(S oldMappingSource, T oldMappingTarget, List<Step> steps);

}
