package com.xebialabs.deployit.mapper;

import java.io.Serializable;
import java.util.List;

import com.xebialabs.deployit.Change;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.ci.mapping.Mapping;

/**
 * A mapper that can generate steps for added, modified and deleted steps.
 * 
 * @param <S>
 *            the type of the source artifact/resource.
 * @param <M>
 *            the type of the mapping.
 * @param <T>
 *            the type of the target middleware.
 */
public abstract class StepGeneratingMapper<S extends Serializable, M extends Mapping, T extends Serializable> extends Mapper<S, M, T> {

	/**
	 * Creates a <tt>StepGeneratingMapper</tt> for the specified deployment change.
	 * 
	 * @param change
	 *            the deployment change for which to calculate mappings.
	 * @param applyDefaultMappings
	 *            default mappings will be applied iff <code>true</code>
	 */
	public StepGeneratingMapper(Change<Deployment> change, boolean applyDefaultMappings) {
		super(change, applyDefaultMappings);
	}

	/**
	 * Creates a <tt>StepGeneratingMapper</tt> for the specified deployment change that will apply default mappings.
	 * 
	 * @param change
	 *            the deployment change for which to calculate mappings.
	 */
	public StepGeneratingMapper(Change<Deployment> change) {
		super(change);
	}

	/**
	 * Generates all the steps that will add artifacts/resources to the middleware.
	 * 
	 * @param steps
	 *            the list of steps to which the generated steps are to be added
	 */
	@SuppressWarnings("unchecked")
	public final void generateAdditionSteps(List<Step> steps) {
		for (M m : addedMappings) {
			generateAdditionStepsForAddedMapping((S) m.getSource(), m, (T) m.getTarget(), steps);
		}
		for (Pair<M, M> m : modifiedMappings) {
			M newM = m.getSecond();
			generateAdditionStepsForModifiedMapping((S) newM.getSource(), newM, (T) newM.getTarget(), steps);
		}
	}

	/**
	 * Generates all the steps that will delete artifacts/resources from the middleware.
	 * 
	 * @param steps
	 *            the list of steps to which the generated steps are to be added
	 */
	@SuppressWarnings("unchecked")
	public final void generateDeletionSteps(List<Step> steps) {
		for (Pair<M, M> m : modifiedMappings) {
			M oldM = m.getFirst();
			generateDeletionStepsForModifiedMapping((S) oldM.getSource(), oldM, (T) oldM.getTarget(), steps);
		}
		for (M m : deletedMappings) {
			generateDeletionStepsForDeletedMapping((S) m.getSource(), m, (T) m.getTarget(), steps);
		}
	}

	/**
	 * Must be implemented by a subclass to generate steps that add an artifact/a resource to the middleware when a
	 * mapping is added.
	 * 
	 * @param newMappingSource
	 *            the source of the new mapping. The value is identical to <tt>newMapping.getSource()</tt> but it is
	 *            provided separately so that the implementor of this method does not have to cast it.
	 * @param newMapping
	 *            the new mapping
	 * @param newMappingTarget
	 *            the target of the new mapping. The value is identical to <tt>newMapping.getTarget()</tt> but it is
	 *            provided separately so that the implementor of this method does not have to cast it.
	 * @param steps
	 *            the list of steps to which the generated steps are to be added
	 */
	protected abstract void generateAdditionStepsForAddedMapping(S newMappingSource, M newMapping, T newMappingTarget, List<Step> steps);

	/**
	 * Can be overridden by a subclass to generate steps that add an artifact/a resource to the middleware when a
	 * mapping is modified. The default implementation invokes
	 * {@link #generateAdditionStepsForAddedMapping(Serializable, Mapping, Serializable, List)}.
	 * 
	 * @param newMappingSource
	 *            the new source of the mapping. The value is identical to
	 *            <tt>newVersionOfModifiedMapping.getSource()</tt> but it is provided separately so that the implementor
	 *            of this method does not have to cast it.
	 * @param newVersionOfModifiedMapping
	 *            the new mapping
	 * @param newMappingTarget
	 *            the new target of the mapping. The value is identical to
	 *            <tt>newVersionOfModifiedMapping.getTarget()</tt> but it is provided separately so that the implementor
	 *            of this method does not have to cast it.
	 * @param steps
	 *            the list of steps to which the generated steps are to be added
	 */
	protected void generateAdditionStepsForModifiedMapping(S newMappingSource, M newVersionOfModifiedMapping, T newMappingTarget, List<Step> steps) {
		generateAdditionStepsForAddedMapping(newMappingSource, newVersionOfModifiedMapping, newMappingTarget, steps);
	}

	/**
	 * Can be overridden by a subclass to generate steps that delete an artifact/a resource from the middleware when a
	 * mapping is modified. The default implementation invokes
	 * {@link #generateDeletionStepsForModifiedMapping(Serializable, Mapping, Serializable, List)}.
	 * 
	 * @param oldMappingSource
	 *            the old source of the modified mapping. The value is identical to
	 *            <tt>oldVersionOfModifiedMapping.getSource()</tt> but it is provided separately so that the implementor
	 *            of this method does not have to cast it.
	 * @param oldVersionOfModifiedMapping
	 *            the old mapping
	 * @param oldMappingTarget
	 *            the old target of the modified mapping. The value is identical to
	 *            <tt>oldVersionOfModifiedMapping.getTarget()</tt> but it is provided separately so that the implementor
	 *            of this method does not have to cast it.
	 * @param steps
	 *            the list of steps to which the generated steps are to be added
	 */
	protected void generateDeletionStepsForModifiedMapping(S oldMappingSource, M oldVersionOfModifiedMapping, T oldMappingTarget, List<Step> steps) {
		generateDeletionStepsForDeletedMapping(oldMappingSource, oldVersionOfModifiedMapping, oldMappingTarget, steps);
	}

	/**
	 * Must be implemented by a subclass to generate steps that delete an artifact/a resource from the middleware when a
	 * mapping is deleted.
	 * 
	 * @param oldMappingSource
	 *            the source of the old mapping. The value is identical to <tt>oldMapping.getSource()</tt> but it is
	 *            provided separately so that the implementor of this method does not have to cast it.
	 * @param oldVersionOfModifiedMapping
	 *            the old mapping
	 * @param oldMappingTarget
	 *            the target of the old mapping. The value is identical to <tt>oldMapping.getTarget()</tt> but it is
	 *            provided separately so that the implementor of this method does not have to cast it.
	 * @param steps
	 *            the list of steps to which the generated steps are to be added
	 */
	protected abstract void generateDeletionStepsForDeletedMapping(S oldMappingSource, M oldVersionOfModifiedMapping, T oldMappingTarget, List<Step> steps);

}