package com.xebialabs.deployit.reflect;

import static com.google.common.base.Preconditions.checkNotNull;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;

import org.apache.log4j.Logger;

import com.xebialabs.deployit.ConfigurationItem;

/**
 * Creates {@link ConfigurationItemDescriptor}s for types.
 */
public class ConfigurationItemDescriptorFactory {

	private static AtomicReference<ConfigurationItemDescriptorFactoryHelper> INSTANCE = new AtomicReference<ConfigurationItemDescriptorFactoryHelper>();

	private static Map<Class<?>, ConfigurationItemDescriptor> cachedDescriptors = new ConcurrentHashMap<Class<?>, ConfigurationItemDescriptor>();

	/**
	 * Used by Spring. Do not invoke!
	 */
	public ConfigurationItemDescriptorFactory(ConfigurationItemDescriptorFactoryHelper instance) {
		if (INSTANCE.getAndSet(checkNotNull(instance)) != null) {
			log.warn("Attempted to reinitialize holder with " + instance);
		}
	}

	/**
	 * Gets the descriptor for a certain configuration item.
	 * 
	 * @param configurationItem
	 *            the configuration item for which to get the descriptor.
	 * @return the descriptor or <tt>null</tt> if no descriptor can be found for the configuration item.
	 */
	public static ConfigurationItemDescriptor getDescriptor(Serializable configurationItem) {

		ConfigurationItemDescriptor configurationItemDescriptor;
		if (INSTANCE.get() != null) {
			configurationItemDescriptor = INSTANCE.get().getDescriptor(configurationItem);
		} else {
			// TODO: Get defaults! and move code to other class
			Class<?> configurationItemClass;
			if (configurationItem instanceof ConfigurationItemProxy) {
				configurationItemClass = ((ConfigurationItemProxy) configurationItem).getBackingObject().getClass();
			} else {
				configurationItemClass = configurationItem.getClass();
			}
			if (!cachedDescriptors.containsKey(configurationItemClass) && (configurationItemClass.getAnnotation(ConfigurationItem.class) != null)) {
				cachedDescriptors.put(configurationItemClass, new ConfigurationItemDescriptor(configurationItemClass, new HashMap<String, String>()));
			}
			configurationItemDescriptor = cachedDescriptors.get(configurationItemClass);
		}
		return configurationItemDescriptor;
	}

	private final Logger log = Logger.getLogger(ConfigurationItemDescriptorFactory.class);

}
