package com.xebialabs.deployit.steps;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;

import org.springframework.core.io.Resource;

import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.hostsession.HostFile;
import com.xebialabs.deployit.hostsession.HostFileUtils;
import com.xebialabs.deployit.hostsession.HostSession;
import com.xebialabs.deployit.util.TemplateResolver;

/**
 * Decorates a {@link HostFile}, replacing {@code $ ...\} placeholders in the underlying file when it is {@link #get()
 * read}.
 * <p>
 * Assumes the contents of the file can be read into a {@code String}!
 * 
 */
class PlaceholderReplacingHostFile implements HostFile {
	private final HostFile decoratedHostFile;
	private final TemplateResolver templateResolver;
	private final boolean lenientResolution;
	private final String resolvedTemplate;

	/**
	 * Creates a {@code PlaceholderReplacingHostFile} with (default) strict resolution.
	 * 
	 * @param decoratedHostFile
	 *            the underlying host file to decorate. May <u>not</u> be a directory or file that cannot be read into a
	 *            string.
	 * @param templateResolver
	 *            the template resolver to use
	 * @see #PlaceholderReplacingHostFile(HostFile, TemplateResolver, boolean)
	 */
	public PlaceholderReplacingHostFile(HostFile decoratedHostFile, TemplateResolver templateResolver) {
		// by default, resolves strictly (i.e. fails if values for some
		// placeholders are missing)
		this(decoratedHostFile, templateResolver, false);
	}

	/**
	 * Creates a {@code PlaceholderReplacingHostFile} with the given resolution strategy.
	 * 
	 * @param decoratedHostFile
	 *            the underlying host file to decorate. May <u>not</u> be a directory or file that cannot be read into a
	 *            string.
	 * @param templateResolver
	 *            the template resolver to use
	 * @see #PlaceholderReplacingHostFile(HostFile, TemplateResolver)
	 */
	public PlaceholderReplacingHostFile(HostFile decoratedHostFile, TemplateResolver templateResolver, boolean lenientResolution) {
		this.decoratedHostFile = checkNotNull(decoratedHostFile);
		checkArgument(!decoratedHostFile.isDirectory(), "Placeholder replacement is only supported on files");

		this.templateResolver = checkNotNull(templateResolver);
		this.lenientResolution = lenientResolution;

		this.resolvedTemplate = resolveTemplate();

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#canExecute()
	 */
	public boolean canExecute() throws RuntimeIOException {
		return decoratedHostFile.canExecute();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#canRead()
	 */
	public boolean canRead() throws RuntimeIOException {
		return decoratedHostFile.canRead();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#canWrite()
	 */
	public boolean canWrite() throws RuntimeIOException {
		return decoratedHostFile.canWrite();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#delete()
	 */
	public boolean delete() throws RuntimeIOException {
		return decoratedHostFile.delete();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#deleteRecursively()
	 */
	public boolean deleteRecursively() throws RuntimeIOException {
		return decoratedHostFile.deleteRecursively();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#deleteRecursively(java.lang .String)
	 */
	public boolean deleteRecursively(String exclusion) throws RuntimeIOException {
		return decoratedHostFile.deleteRecursively(exclusion);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#exists()
	 */
	public boolean exists() throws RuntimeIOException {
		return decoratedHostFile.exists();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#get()
	 */
	public InputStream get() throws RuntimeIOException {
		return new ByteArrayInputStream(resolvedTemplate.getBytes());
	}

	private String resolveTemplate() {
		String template;

		// will be a problem with large files!
		template = HostFileUtils.getHostFileAsString(decoratedHostFile);

		assert (template != null);

		String resolved = (lenientResolution ? templateResolver.resolveLenient(template) : templateResolver.resolveStrict(template));
		return resolved;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#get(java.io.OutputStream)
	 */
	public void get(OutputStream out) throws RuntimeIOException {
		// assumes that the implementation of get(out) delegates to get()!
		decoratedHostFile.get(out);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#get(java.io.File)
	 */
	public void get(File file) throws RuntimeIOException {
		// assumes that the implementation of get(out) delegates to get()!
		decoratedHostFile.get(file);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#getFile(java.lang.String)
	 */
	public HostFile getFile(String name) throws RuntimeIOException {
		return decoratedHostFile.getFile(name);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#getName()
	 */
	public String getName() {
		return decoratedHostFile.getName();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#getParent()
	 */
	public String getParent() {
		return decoratedHostFile.getParent();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#getParentFile()
	 */
	public HostFile getParentFile() {
		return decoratedHostFile.getParentFile();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#getPath()
	 */
	public String getPath() {
		return decoratedHostFile.getPath();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#getSession()
	 */
	public HostSession getSession() {
		return decoratedHostFile.getSession();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#isDirectory()
	 */
	public boolean isDirectory() throws RuntimeIOException {
		return decoratedHostFile.isDirectory();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#length()
	 */

	public long length() throws RuntimeIOException {
		return resolvedTemplate.length();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#list()
	 */
	public List<String> list() throws RuntimeIOException {
		return decoratedHostFile.list();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#listFiles()
	 */
	public List<HostFile> listFiles() throws RuntimeIOException {
		return decoratedHostFile.listFiles();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#mkdir()
	 */
	public void mkdir() throws RuntimeIOException {
		decoratedHostFile.mkdir();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#mkdirs()
	 */
	public void mkdirs() throws RuntimeIOException {
		decoratedHostFile.mkdirs();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#put(long)
	 */
	public OutputStream put(long length) throws RuntimeIOException {
		return decoratedHostFile.put(length);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#put(java.io.InputStream, long)
	 */
	public void put(InputStream in, long length) throws RuntimeIOException {
		decoratedHostFile.put(in, length);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#put(java.io.File)
	 */
	public void put(File file) throws RuntimeIOException {
		decoratedHostFile.put(file);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.xebia.ad.environment.session.HostFile#put(org.springframework.core .io.Resource)
	 */
	public void put(Resource resource) throws RuntimeIOException {
		decoratedHostFile.put(resource);
	}

	@Override
	public String toString() {
		return decoratedHostFile + " (decorated)";
	}
}
