package com.xebialabs.deployit.util;

import java.util.Collection;

import org.apache.commons.collections.Transformer;
import org.apache.commons.collections.iterators.TransformIterator;
import org.apache.commons.lang.StringUtils;

/**
 * Utility class for some useful String manipulations.
 */
public class ExtendedStringUtils {

	/**
	 * The default join separator used by {@link #join(Collection, Transformer)}.
	 */
	public static final String DEFAULT_JOIN_SEPARATOR = ",";

	/**
	 * Returns the string wrapped between double quotes if it contains a space or if it {@link StringUtils#isEmpty is
	 * empty}.
	 * 
	 * @param str
	 *            the string to quote
	 * @return the quoted string
	 */
	public static String getQuotedString(String str) {
		if (StringUtils.isEmpty(str)) {
			return "";
		} else if (StringUtils.containsAny(str, " \'\"")) {
			return "\"" + str + "\"";
		} else {
			return str;
		}
	}

	/**
	 * Joins the string representations of a collection to form one string using the {@link #DEFAULT_JOIN_SEPARATOR
	 * default join separator}.
	 * 
	 * @param <T>
	 *            the type of the collection members.
	 * @param collection
	 *            the collection to join.
	 * @param toStringGenerator
	 *            the {@link Transformer} that transforms a collection member to its string representation.
	 * @return the joined string.
	 */
	public static <T> String join(Collection<T> collection, Transformer toStringGenerator) {
		return join(collection, toStringGenerator, DEFAULT_JOIN_SEPARATOR);
	}

	/**
	 * Joins the string representations of a collection to form one string.
	 * 
	 * @param <T>
	 *            the type of the collection members.
	 * @param collection
	 *            the collection to join.
	 * @param toStringGenerator
	 *            the {@link Transformer} that transforms a collection member to its string representation.
	 * @param separator
	 *            the string that will separate the strings when they are joined together.
	 * @return the joined string.
	 */
	public static <T> String join(Collection<T> collection, final Transformer toStringGenerator, String separator) {
		/*
		 * OK, this code is a *little* cryptic, I admit ;-) Essentially, what's going on is: 1) join will call toString
		 * on every item in the collection passed to it 2) we want to replace the standard toString of the collection
		 * items with a call to the toStringGenerator 3) so, we transform the collection, replacing every item with an
		 * object which has no behaviour except an implementation of toString... 4) ...and this toString implementation
		 * simply calls the toStringGenerator on the original item
		 */
		return StringUtils.join(new TransformIterator(collection.iterator(), toStringGenerator), separator);
	}

}
