/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.reflect;

import com.xebialabs.deployit.ConfigurationItem;
import com.xebialabs.deployit.ConfigurationItemProperty;
import com.xebialabs.deployit.ConfigurationItemProperty.Size;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

import java.io.Serializable;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Describes a property of a {@link ConfigurationItem}.
 */
@SuppressWarnings("serial")
public class ConfigurationItemPropertyDescriptor implements Serializable, ConfigurationItemPropertyHolder {
	/*
	 * N.B.: Default constructor should be there for BlazeDS serialization (AS -> Java) to work.
	 * 
	 * N.B.: Getters and setters should both be there for BlazeDS serialization (Java -> AS) to work
	 */

	private ConfigurationItemDescriptor owningConfigurationItemDescriptor;

	private ConfigurationItemPropertyDescriptor owningPropertyDescriptor;

	private String name;

	private ConfigurationItemPropertyType type;

	private boolean asContainment;

	private String[] enumValues;

	private String collectionMemberClassname;

	private String propertyClassname;

	private ConfigurationItemPropertyDescriptor[] listObjectPropertyDescriptors;

	private String label;

	private String description;

	private boolean required;

	private boolean editable;

	private boolean password;

	private boolean identifying;

	private boolean discoveryParam;

	private boolean discoveryRequired;

	private String category;

	private String defaultValue;

	private ConfigurationItemProperty.Size size;

	// not sent to the clients
	private transient Field propertyField;

	// not sent to the clients
	private transient Class<?> propertyClass;

	// not sent to the clients
	private transient Class<?> collectionMemberClass;

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public ConfigurationItemPropertyDescriptor() {
	}

	/**
	 * Constructs a <tt>ConfigurationItemPropertyDescriptor</tt>. Do not invoke!
	 */
	public ConfigurationItemPropertyDescriptor(ConfigurationItemDescriptor owningCIDescriptor, ConfigurationItemPropertyDescriptor owningPropertyDescriptor,
	        Field field, String defaultValue, boolean isTopLevelClass) {
		this.owningConfigurationItemDescriptor = owningCIDescriptor;
		this.owningPropertyDescriptor = owningPropertyDescriptor;
		setPropertyField(field);
		ConfigurationItemProperty propertyAnnotation = field.getAnnotation(ConfigurationItemProperty.class);
		initNames(propertyAnnotation);
		initType(propertyAnnotation, isTopLevelClass);
		initValidation(propertyAnnotation, defaultValue);
		initCapabilities(propertyAnnotation);
	}

	/**
	 * Constructs a list of <tt>ConfigurationItemPropertyDescriptor</tt>s. Do not invoke!
	 */
	public static ConfigurationItemPropertyDescriptor[] getConfigurationItemPropertyDescriptorsForListObject(ConfigurationItemDescriptor owningCIDescriptor,
	        ConfigurationItemPropertyDescriptor owningPropertyDescriptor, Class<?> listObjectClass) {
		return getConfigurationItemPropertyDescriptors(owningCIDescriptor, owningPropertyDescriptor, listObjectClass, null, null, false);
	}

	/**
	 * Constructs a list of <tt>ConfigurationItemPropertyDescriptor</tt>s. Do not invoke!
	 */
	public static ConfigurationItemPropertyDescriptor[] getConfigurationItemPropertyDescriptorsForTopLevelClass(ConfigurationItemDescriptor owningCIDescriptor,
	        Class<?> typeClass, Map<String, String> configuredDefaults, String defaultValueKeyPrefix) {
		return getConfigurationItemPropertyDescriptors(owningCIDescriptor, null, typeClass, configuredDefaults, defaultValueKeyPrefix, true);
	}

	private static ConfigurationItemPropertyDescriptor[] getConfigurationItemPropertyDescriptors(ConfigurationItemDescriptor owningCIDescriptor,
	        ConfigurationItemPropertyDescriptor owningPropertyDescriptor, Class<?> typeClass, Map<String, String> configuredDefaults,
	        String defaultValueKeyPrefix, boolean isTopLevelClass) {
		List<ConfigurationItemPropertyDescriptor> propertyDescriptorList = new ArrayList<ConfigurationItemPropertyDescriptor>();
		for (Field field : getAllAnnotatedProperties(typeClass)) {
			String propertyName = field.getName();
			String defaultValue;
			if (configuredDefaults != null && defaultValueKeyPrefix != null) {
				String defaultValueKey = defaultValueKeyPrefix + propertyName.toLowerCase();
				defaultValue = configuredDefaults.get(defaultValueKey);
			} else {
				defaultValue = null;
			}

			propertyDescriptorList.add(new ConfigurationItemPropertyDescriptor(owningCIDescriptor, owningPropertyDescriptor, field, defaultValue,
			        isTopLevelClass));
		}
		return propertyDescriptorList.toArray(new ConfigurationItemPropertyDescriptor[propertyDescriptorList.size()]);
	}

	private static List<Field> getAllAnnotatedProperties(Class<?> ciClass) {
		return getAllAnnotatedProperties(ciClass, ConfigurationItemProperty.class);
	}

	private void initNames(ConfigurationItemProperty fieldAnnotation) {
		name = propertyField.getName();
		label = fieldAnnotation.label();
		if (StringUtils.isBlank(label)) {
			label = StringUtils.capitalize(StringUtils.join(StringUtils.splitByCharacterTypeCamelCase(name), " "));
		}
		description = fieldAnnotation.description();
		if (StringUtils.isBlank(description)) {
			description = label;
		}
	}

	private void initType(final ConfigurationItemProperty fieldAnnotation, final boolean isTopLevelClass) {
		propertyClass = propertyField.getType();
		propertyClassname = propertyClass.getName();

		if (propertyClass.equals(boolean.class)) {
			initBooleanType();
		} else if (propertyClass.equals(int.class)) {
			initIntegerType();
		} else if (propertyClass.equals(String.class)) {
			initStringType();
		} else if (propertyClass.isEnum()) {
			initEnumType(propertyClass);
		} else if (propertyClass.equals(List.class)) {
			initListType(propertyField, isTopLevelClass);
		} else if (propertyClass.equals(Set.class)) {
			initSetType(fieldAnnotation);
		} else if (Serializable.class.isAssignableFrom(propertyClass)) {
			initCIType(fieldAnnotation);
		} else {
			throw new IllegalArgumentException("Property " + propertyField.getDeclaringClass().getName() + "." + propertyField.getName()
			        + " is of an unsupported type");
		}
	}

	private void initUnsupportedCiType() {
		type = ConfigurationItemPropertyType.UNSUPPORTED;
	}

	private void initBooleanType() {
		type = ConfigurationItemPropertyType.BOOLEAN;
	}

	private void initIntegerType() {
		type = ConfigurationItemPropertyType.INTEGER;
	}

	private void initStringType() {
		type = ConfigurationItemPropertyType.STRING;
	}

	private void initEnumType(Class<?> fieldType) {
		type = ConfigurationItemPropertyType.ENUM;
		getEnumValues(fieldType);
	}

	private void getEnumValues(Class<?> fieldType) {
		List<String> enumVals = new ArrayList<String>();
		for (Object each : fieldType.getEnumConstants()) {
			enumVals.add(each.toString());
		}
		enumValues = enumVals.toArray(new String[enumVals.size()]);
	}

	private void initListType(final Field f, final boolean isTopLevelClass) {
		if (!isTopLevelClass) {
			throw new IllegalArgumentException("Property " + propertyField.getDeclaringClass().getName() + "." + propertyField.getName()
			        + " is a list of objects in a list of objects");
		}

		type = ConfigurationItemPropertyType.LIST_OF_OBJECTS;

		collectionMemberClass = getListObjectClass(f);
		collectionMemberClassname = collectionMemberClass.getName();
		listObjectPropertyDescriptors = ConfigurationItemPropertyDescriptor.getConfigurationItemPropertyDescriptorsForListObject(
		        owningConfigurationItemDescriptor, this, collectionMemberClass);
		if (listObjectPropertyDescriptors.length == 0) {
			throw new IllegalArgumentException("The list object " + collectionMemberClassname + " of " + f.getDeclaringClass().getName() + "." + f.getName()
			        + " does not contain any @" + ConfigurationItemProperty.class.getSimpleName() + " annotations");
		}
	}

	private void initSetType(final ConfigurationItemProperty fieldAnnotation) {
		java.lang.reflect.Type genericType = propertyField.getGenericType();
		if (genericType instanceof ParameterizedType) {
			ParameterizedType parameterizedType = (ParameterizedType) genericType;

			for (Type t : parameterizedType.getActualTypeArguments()) {
				collectionMemberClass = (Class<?>) t;
				collectionMemberClassname = collectionMemberClass.getName();
			}
			if (String.class.isAssignableFrom(collectionMemberClass)) {
				initSetOfStringsType();
			} else if (Serializable.class.isAssignableFrom(collectionMemberClass)) {
				initSetOfCIsType(fieldAnnotation);
			} else {
				logger.error("The property (Set) " + propertyField.getName() + " is UNSUPPORTED");
				initUnsupportedCiType();
			}
		}
	}

	private void initSetOfStringsType() {
		type = ConfigurationItemPropertyType.SET_OF_STRINGS;
	}

	private void initSetOfCIsType(final ConfigurationItemProperty fieldAnnotation) {
		type = ConfigurationItemPropertyType.SET_OF_CIS;
		asContainment = fieldAnnotation.asContainment();
	}

	private void initCIType(final ConfigurationItemProperty fieldAnnotation) {
		type = ConfigurationItemPropertyType.CI;
		asContainment = fieldAnnotation.asContainment();
	}

	private Class<?> getListObjectClass(Field f) {
		ParameterizedType listType = (ParameterizedType) f.getGenericType();
		Type[] listTypeArguments = listType.getActualTypeArguments();
		if (listTypeArguments.length != 1) {
			throw new IllegalArgumentException(f.getDeclaringClass().getName() + "." + f.getName()
			        + "is of type java.util.List but does not have exactly 1 type parameter");
		}
		if (!(listTypeArguments[0] instanceof Class)) {
			throw new IllegalArgumentException(f.getDeclaringClass().getName() + "." + f.getName()
			        + "is of type java.util.List but the first type parameter is not a concrete class");
		}
		return (Class<?>) listTypeArguments[0];
	}

	private void initValidation(ConfigurationItemProperty fieldAnnotation, String defaultValue) {
		required = fieldAnnotation.required();
		this.defaultValue = defaultValue;
	}

	private void initCapabilities(ConfigurationItemProperty fieldAnnotation) {
		password = fieldAnnotation.password();
		editable = fieldAnnotation.editable();
		identifying = fieldAnnotation.identifying();
		discoveryParam = fieldAnnotation.discoveryParam();
		discoveryRequired = fieldAnnotation.discoveryRequired();
		category = fieldAnnotation.category();
		size = fieldAnnotation.size();
		if (size == Size.DEFAULT) {
			if (type == ConfigurationItemPropertyType.BOOLEAN || type == ConfigurationItemPropertyType.INTEGER) {
				size = Size.SMALL;
			} else {
				size = Size.MEDIUM;
			}
		}
	}

	/**
	 * Gets a property from a configuration item.
	 * 
	 * @param configurationItem
	 *            the configuration item to get the property value from.
	 * @return the property value on the configuration item.
	 */
	public Object getPropertyValueFromConfigurationItem(Object configurationItem) {
		configurationItem = ConfigurationItemReflectionUtils.getRealConfigurationItem(configurationItem);
		try {
			return getPropertyField().get(configurationItem);
		} catch (IllegalArgumentException exc) {
			throw new RuntimeException("Cannot read property " + getName() + " from configuration item of type "
			        + getOwningConfigurationItemDescriptor().getType(), exc);
		} catch (IllegalAccessException exc) {
			throw new RuntimeException("Cannot read property " + getName() + " from configuration item of type "
			        + getOwningConfigurationItemDescriptor().getType(), exc);
		}
	}

	/**
	 * Sets a property value on a configuration item.
	 * 
	 * @param configurationItem
	 *            the configuration item to set the value on.
	 * @param value
	 *            the value to set
	 */
	public void setPropertyValueInConfigurationItem(Object configurationItem, Object value) {
		configurationItem = ConfigurationItemReflectionUtils.getRealConfigurationItem(configurationItem);
		try {
			getPropertyField().set(configurationItem, value);
		} catch (IllegalArgumentException exc) {
			throw new RuntimeException("Cannot write property " + getName() + " from configuration item of type "
			        + getOwningConfigurationItemDescriptor().getType(), exc);
		} catch (IllegalAccessException exc) {
			throw new RuntimeException("Cannot write property " + getName() + " from configuration item of type "
			        + getOwningConfigurationItemDescriptor().getType(), exc);
		}
	}

	/**
	 * Gets the descriptor of the configuration item that has this property.
	 * 
	 * @return the descriptor of the configuration item that has this property.
	 */
	public ConfigurationItemDescriptor getOwningConfigurationItemDescriptor() {
		return owningConfigurationItemDescriptor;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setOwningConfigurationItemDescriptor(ConfigurationItemDescriptor configurationItemDescriptor) {
		this.owningConfigurationItemDescriptor = configurationItemDescriptor;
	}

	/**
	 * Gets the descriptor of the property that has this property.
	 * 
	 * @return the descriptor of the property that has this property.
	 */
	public ConfigurationItemPropertyDescriptor getOwningPropertyDescriptor() {
		return owningPropertyDescriptor;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setOwningPropertyDescriptor(ConfigurationItemPropertyDescriptor owningPropertyDescriptor) {
		this.owningPropertyDescriptor = owningPropertyDescriptor;
	}

	/**
	 * Gets the name of this property.
	 * 
	 * @return the name of this property.
	 */
	public String getName() {
		return name;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * Gets the {@link Field} for this property.
	 * 
	 * @return the field for this property.
	 */
	public Field getPropertyField() {
		if (propertyField == null) {
			try {
				Field field;
				if (owningPropertyDescriptor != null) {
					field = getDeclaredFieldInClassAndSuperClasses(owningPropertyDescriptor.getCollectionMemberClass(), name);
				} else {
					field = getDeclaredFieldInClassAndSuperClasses(owningConfigurationItemDescriptor.getTypeClass(), name);
				}
				setPropertyField(field);
			} catch (NoSuchFieldException exc) {
				throw new RuntimeException("Cannot find field " + name + " in " + owningConfigurationItemDescriptor.getTypeClass(), exc);
			}
		}
		return propertyField;
	}

	private void setPropertyField(Field propertyField) {
		this.propertyField = propertyField;
		this.propertyField.setAccessible(true);
	}

	/**
	 * Gets the type of this property.
	 * 
	 * @return the type of this property.
	 */
	public ConfigurationItemPropertyType getType() {
		return type;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setType(ConfigurationItemPropertyType type) {
		this.type = type;
	}

	/**
	 * Gets the class name of this property
	 * 
	 * @return the class name of this property
	 */
	public String getPropertyClassname() {
		return propertyClassname;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setPropertyClassname(String propertyClassname) {
		this.propertyClassname = propertyClassname;
	}

	/**
	 * Gets the class of this property.
	 * 
	 * @return the class of this property.
	 */
	public Class<?> getPropertyClass() {
		if (propertyClass == null) {
			if (propertyClassname == null) {
				return null;
			}

			try {
				if (propertyClassname.equals(Boolean.TYPE.getName())) {
					propertyClass = Boolean.TYPE;
				} else if (propertyClassname.equals(Integer.TYPE.getName())) {
					propertyClass = Integer.TYPE;
				} else {
					propertyClass = Class.forName(propertyClassname);
				}
			} catch (ClassNotFoundException exc) {
				throw new RuntimeException("Unable to get class for name " + propertyClassname, exc);
			}
		}
		return propertyClass;
	}

	/**
	 * Returns <code>true</code> iff this property is of type {@link ConfigurationItemPropertyType#CI} or {@link ConfigurationItemPropertyType#SET_OF_CIS} and
	 * it is modeled as parent/child containment in the JCR tree.
	 * 
	 * @return <code>true</code> iff this property is modeled as containment
	 */
	public boolean asContainment() {
		return asContainment;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setAsContainment(boolean asContainment) {
		this.asContainment = asContainment;
	}

	/**
	 * Gets the enum values, if this property is an {@link ConfigurationItemPropertyType#ENUM}
	 * 
	 * @return the enum values or <tt>null</tt> if this property is not an {@link ConfigurationItemPropertyType#ENUM},
	 */
	public String[] getEnumValues() {
		return enumValues;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setEnumValues(String[] enumValues) {
		this.enumValues = enumValues;
	}

	/**
	 * Gets the collection member class name, if this property is an {@link ConfigurationItemPropertyType#LIST_OF_OBJECTS}
	 * 
	 * @return the collection member class name or <tt>null</tt> if this property is not an {@link ConfigurationItemPropertyType#LIST_OF_OBJECTS},
	 */
	public String getCollectionMemberClassname() {
		return collectionMemberClassname;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setCollectionMemberClassname(String collectionMemberClassname) {
		this.collectionMemberClassname = collectionMemberClassname;
	}

	/**
	 * Gets the collection member class, if this property is an {@link ConfigurationItemPropertyType#LIST_OF_OBJECTS}
	 * 
	 * @return the collection member class or <tt>null</tt> if this property is not an {@link ConfigurationItemPropertyType#LIST_OF_OBJECTS}.
	 */
	public Class<?> getCollectionMemberClass() {
		if (collectionMemberClass == null) {
			if (collectionMemberClassname == null) {
				return null;
			}

			try {
				collectionMemberClass = Class.forName(collectionMemberClassname);
			} catch (ClassNotFoundException exc) {
				throw new RuntimeException("Unable to get class for name " + collectionMemberClassname, exc);
			}
		}
		return collectionMemberClass;
	}

	/**
	 * Gets the property descriptors of the collection member classes, if this property is an {@link ConfigurationItemPropertyType#LIST_OF_OBJECTS}
	 * 
	 * @return the property descriptors of the collection member classes or <tt>null</tt> if this property is not an
	 *         {@link ConfigurationItemPropertyType#LIST_OF_OBJECTS}.
	 */
	public ConfigurationItemPropertyDescriptor[] getListObjectPropertyDescriptors() {
		return listObjectPropertyDescriptors;
	}

	/**
	 * @see #getListObjectPropertyDescriptors().
	 */
	public ConfigurationItemPropertyDescriptor[] getPropertyDescriptors() {
		return getListObjectPropertyDescriptors();
	}

	/**
	 * Gets the property descriptor for a named property of the collection member class, if this property is an
	 * {@link ConfigurationItemPropertyType#LIST_OF_OBJECTS}
	 * 
	 * @param propertyName
	 *            the name of the property sought after.
	 * @return the property descriptor for a named property of the collection member class or <tt>null</tt> if this property is not an
	 *         {@link ConfigurationItemPropertyType#LIST_OF_OBJECTS}.
	 */
	public ConfigurationItemPropertyDescriptor getPropertyDescriptor(String propertyName) {
		for (ConfigurationItemPropertyDescriptor each : getPropertyDescriptors()) {
			if (each.getName().equals(propertyName)) {
				return each;
			}
		}
		return null;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setListObjectPropertyDescriptors(ConfigurationItemPropertyDescriptor[] listObjectPropertyDescriptors) {
		this.listObjectPropertyDescriptors = listObjectPropertyDescriptors;
	}

	/**
	 * Gets the label of this property.
	 * 
	 * @return the label of this property.
	 */
	public String getLabel() {
		return label;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setLabel(String label) {
		this.label = label;
	}

	/**
	 * Gets the description of this property.
	 * 
	 * @return the description of this property.
	 */
	public String getDescription() {
		return description;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setDescription(String description) {
		this.description = description;
	}

	/**
	 * Return whether this property is required.
	 * 
	 * @return <tt>true</tt> iff this property is required.
	 */
	public boolean isRequired() {
		return required;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setRequired(boolean required) {
		this.required = required;
	}

	/**
	 * Return whether this property is a password.
	 * 
	 * @return <tt>true</tt> iff this property is a password.
	 */
	public boolean isPassword() {
		return password;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setPassword(boolean password) {
		this.password = password;
	}

	/**
	 * Return whether this property is a editable.
	 * 
	 * @return <tt>true</tt> iff this property is a editable.
	 */
	public boolean isEditable() {
		return editable;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setEditable(boolean editable) {
		this.editable = editable;
	}

	/**
	 * Return whether this property is an identifying property.
	 * 
	 * @return <tt>true</tt> iff this property is an identifying property.
	 */
	public boolean isIdentifying() {
		return identifying;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setIdentifying(boolean identifying) {
		this.identifying = identifying;
	}

	/**
	 * Returns whether this property should be shown as an editable field when discovering a new instance of this configuration item.
	 * 
	 * @return <tt>true</tt> iff this property is a parameter for discovery.
	 */
	public boolean isDiscoveryParam() {
		return discoveryParam;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setDiscoveryParam(boolean discoveryParam) {
		this.discoveryParam = discoveryParam;
	}

	public boolean isDiscoveryRequired() {
		return discoveryRequired;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setDiscoveryRequired(boolean discoveryRequired) {
		this.discoveryRequired = discoveryRequired;
	}

	public String getDefaultValue() {
		return defaultValue;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setDefaultValue(String defaultValue) {
		this.defaultValue = defaultValue;
	}

	/**
	 * Gets the category of this property.
	 * 
	 * @return the category of this property.
	 */
	public String getCategory() {
		return category;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setCategory(String category) {
		this.category = category;
	}

	/**
	 * Gets the display size of this property.
	 * 
	 * @return the display size of this property.
	 */
	public ConfigurationItemProperty.Size getSize() {
		return size;
	}

	/**
	 * @deprecated Needed for BlazeDS. Do not invoke!
	 */
	public void setSize(ConfigurationItemProperty.Size size) {
		this.size = size;
	}

	@Override
	public String toString() {
		return "ConfigurationItemPropertyDescriptor of " + getOwningConfigurationItemDescriptor().getType() + "." + getName();
	}

	static Field getDeclaredFieldInClassAndSuperClasses(Class<? extends Object> theClass, String name) throws NoSuchFieldException {
		try {
			return theClass.getDeclaredField(name);
		} catch (NoSuchFieldException nsfe) {
			Class<? extends Object> superClass = theClass.getSuperclass();
			if (!Object.class.getName().equals(superClass.getName())) {
				return getDeclaredFieldInClassAndSuperClasses(superClass, name);
			} else {
				throw new NoSuchFieldException("Cannot find field " + name + " in " + theClass + " giving up.");
			}
		}
	}

    static Field getSingleAnnotatedProperty(Class<?> ciClass, Class<? extends Annotation> annotationType, String messageIfMoreThanOne) {
        final List<Field> allAnnotatedProperties = getAllAnnotatedProperties(ciClass, annotationType);
        if (allAnnotatedProperties.size() == 1) {
            return allAnnotatedProperties.get(0);
        } else if (allAnnotatedProperties.size() > 1) {
            throw new IllegalArgumentException(messageIfMoreThanOne);
        }
        return null;
    }

	static List<Field> getAllAnnotatedProperties(Class<?> ciClass, Class<? extends Annotation> annotationType) {
		List<Field> annotatedProperties = new ArrayList<Field>();

		// First get annotated properties in our super class
		final Class<?> aClass = ciClass.getSuperclass();
		if (aClass == null)
			return annotatedProperties;

		if (!Object.class.getName().equals(aClass.getName())) {
			annotatedProperties.addAll(getAllAnnotatedProperties(ciClass.getSuperclass(), annotationType));
		}

		// Then get our own annotated properties
		Field[] fields = ciClass.getDeclaredFields();
		for (Field field : fields) {
			Annotation configurationItemProperty = field.getAnnotation(annotationType);
			if (configurationItemProperty != null) {
				annotatedProperties.add(field);
			}
		}
		return annotatedProperties;
	}

	private static Logger logger = Logger.getLogger(ConfigurationItemPropertyDescriptor.class);

}
