package com.xebialabs.deployit.ci.artifact.mapping;

import com.google.common.collect.Lists;
import com.xebialabs.deployit.ConfigurationItemProperty;
import com.xebialabs.deployit.ci.Hosted;
import com.xebialabs.deployit.ci.artifact.Folder;
import com.xebialabs.deployit.ci.mapping.KeyValuePair;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Predicate;
import org.apache.log4j.Logger;

import java.util.ArrayList;
import java.util.List;

/**
 * A mapping of a {@link com.xebialabs.deployit.ci.artifact.Folder} configuration item to a {@link com.xebialabs.deployit.ci.Host}.
 */
@SuppressWarnings("serial")
public class FolderMapping<S extends Folder, T extends Hosted> extends DeployableArtifactMapping<S, T> {

	public static final String TARGET_DIRECTORY_KEY_NAME = "TARGET_DIRECTORY";

	@ConfigurationItemProperty(required = true, description = "The directory on the target host to which the configuration is to be copied. Example: /data/petclinic/1.0/conf. It can be overidden by using Key " + TARGET_DIRECTORY_KEY_NAME)
	private String targetDirectory;

	@ConfigurationItemProperty(description = "create the target directory and sub directory if they are not exist")
	private boolean createIfNotExist = false;

	@ConfigurationItemProperty(description = "The target directory is shared when another resources using it as well, default true.")
	private boolean shared = true;

	public FolderMapping() {

	}

	public FolderMapping(S source, T target, String targetDirectory) {
		super(source,target);
		setTargetDirectory(targetDirectory);
	}

	public FolderMapping(S source, T target, String targetDirectory, boolean shared) {
		super(source,target);
		setTargetDirectory(targetDirectory);
		setShared(shared);
	}


	@Override
	public List<KeyValuePair> getKeyValuePairs() {
		List<KeyValuePair> kvps = super.getKeyValuePairs();
		Object existingTargetDirectoryKvp = CollectionUtils.find(kvps, new Predicate() {
			public boolean evaluate(Object each) {
				return ((KeyValuePair) each).getKey().equals(TARGET_DIRECTORY_KEY_NAME);
			}
		});

		if (existingTargetDirectoryKvp == null) {
			kvps.add(new KeyValuePair(TARGET_DIRECTORY_KEY_NAME, getTargetDirectory()));
		} else {
			logger.warn("ConfigurationFilesMapping " + this + " already contains a key/value pair for key " + TARGET_DIRECTORY_KEY_NAME
					+ ". Not overriding it.");
		}
		return kvps;
	}

	public String getTargetDirectory() {
		return targetDirectory;
	}

	public void setTargetDirectory(String targetDirectory) {
		this.targetDirectory = targetDirectory;
	}

	public boolean isCreateIfNotExist() {
		return createIfNotExist;
	}

	public void setCreateIfNotExist(boolean createIfNotExist) {
		this.createIfNotExist = createIfNotExist;
	}

	public boolean isShared() {
		return shared;
	}

	public void setShared(boolean shared) {
		this.shared = shared;
	}

	private static Logger logger = Logger.getLogger(FolderMapping.class);
}
