/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.mapper;

import com.google.common.base.Function;
import com.qrmedia.commons.lang.ClassUtils;
import com.xebialabs.deployit.Change;
import com.xebialabs.deployit.ResolutionException;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.ci.mapping.KeyValuePair;
import com.xebialabs.deployit.ci.mapping.Mapping;
import com.xebialabs.deployit.ci.mapping.SourcePropertyOverridingMapping;
import com.xebialabs.deployit.reflect.ConfigurationItemReflectionUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Predicate;

import java.io.Serializable;
import java.util.*;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Calculates which source artifact/resource is mapped to what middleware target.
 *
 * @param <S>
 * the type of the source artifact/resource.
 * @param <M>
 * the type of the mapping.
 * @param <T>
 * the type of the target middleware.
 */
@SuppressWarnings("unchecked")
public abstract class Mapper<S extends Serializable, M extends Mapping<? super S, ? super T>, T extends Serializable> {

	protected Change<Deployment> change;

	protected List<M> oldMappings;

	protected List<M> newMappings;

	protected List<M> addedMappings;

	protected List<Pair<M, M>> modifiedMappings;

	protected List<M> deletedMappings;

	private Class<S> sourceCiClass;

	private Class<M> mappingCiClass;

	private Class<T> targetCiClass;

	private boolean applyDefaultMappings;

	private HashSet<T> affectedTargets;

	/**
	 * Creates a <tt>Mapper</tt> for the specified deployment change. Uses <a
	 * href="http://blog.xebia.com/2009/02/07/acessing-generic-types-at-runtime-in-java/">the approach described in this blog by Arjan Blokzijl</a> to infer the
	 * actual type arguments.
	 *
	 * @param change               the deployment change for which to calculcate mappings.
	 * @param applyDefaultMappings default mappings will be applied iff <code>true</code>
	 */
	protected Mapper(Change<Deployment> change, boolean applyDefaultMappings) {
		this.change = change;
		this.applyDefaultMappings = applyDefaultMappings;
		@SuppressWarnings("rawtypes")
		List<Class<?>> actualTypeArguments = ClassUtils.getActualTypeArguments((Class<? extends Mapper>) getClass(), Mapper.class);
		if (actualTypeArguments.size() != 3) {
			throw new IllegalArgumentException("Not exactly three parametrized types?");
		}
		sourceCiClass = (Class<S>) checkNotNull(actualTypeArguments.get(0));
		mappingCiClass = (Class<M>) checkNotNull(actualTypeArguments.get(1));
		targetCiClass = (Class<T>) checkNotNull(actualTypeArguments.get(2));
		init();
	}

	/**
	 * Creates a <tt>Mapper</tt> for the specified deployment change that will <em>NOT</em> apply default mappings.
	 *
	 * @param change the deployment change for which to calculcate mappings.
	 */
	protected Mapper(Change<Deployment> change) {
		this(change, false);
	}

	protected void init() {
		oldMappings = calculateMappings(change.getOldRevision());
		newMappings = calculateMappings(change.getNewRevision());

		addedMappings = substractSimilarMappings(newMappings, oldMappings);
		modifiedMappings = findModifiedMappings(oldMappings, newMappings);
		deletedMappings = substractSimilarMappings(oldMappings, newMappings);
	}

	/**
	 * Returns the list of old mappings. A mapping is considered to be old if it was part of the old revision of the deployment.
	 *
	 * @return a list of the old mappings
	 */
	public List<M> getOldMappings() {
		return oldMappings;
	}

	/**
	 * Returns the list of new mappings. A mapping is considered to be new if it was part of the new revision of the deployment.
	 *
	 * @return a list of the new mappings
	 */
	public List<M> getNewMappings() {
		return newMappings;
	}

	/**
	 * Returns the list of added mappings. A mapping is considered to be added for one of two reasons:
	 * <ul>
	 * <li>If the deployment was added, all its mappings will be considered to be added.</li>
	 * <li>If a deployment was modified and this mapping was added to it..</li>
	 * </ul>
	 *
	 * @return a list of the added mappings
	 */
	public List<M> getAddedMappings() {
		return addedMappings;
	}

	/**
	 * Returns the list of modified mappings. A mapping is considered to be modified if the deployment was modified and a
	 * {@link ConfigurationItemReflectionUtils#isSimilar(Serializable, Serializable) similar} but not
	 * {@link ConfigurationItemReflectionUtils#isIdentical(Serializable, Serializable)} mapping existed in the old and the new version.
	 *
	 * @return list of modified mappings as {@link Pair pairs}; the {@link Pair#getFirst() first element of the pair} is the old mapping and the
	 *         {@link Pair#getSecond() second element of the pair} is the new mapping.
	 */
	public List<Pair<M, M>> getModifiedMappings() {
		return modifiedMappings;
	}

	/**
	 * Returns the list of deleted mappings. A mapping is considered to be deleted for one of two reasons:
	 * <ul>
	 * <li>If the deployment was deleted, all its mappings will be considered to be deleted.</li>
	 * <li>If a deployment was modified and this mapping was deleted from it.</li>
	 * </ul>
	 *
	 * @return a list of the added mappings
	 */
	public List<M> getDeletedMappings() {
		return deletedMappings;
	}

	/**
	 * Gets all the targets of all the mappings, old and new.
	 *
	 * @return a set of all targets.
	 */
	public Set<T> getAllTargets() {
		HashSet<T> allTargets = new HashSet<T>();
		gatherTargets(oldMappings, allTargets);
		gatherTargets(newMappings, allTargets);
		return allTargets;
	}

	/**
	 * Gets the affected targets of all the mappings. The affected targets are targets for which one ore more mappings were added, modified or deleted.
	 *
	 * @return a set of the affected targets.
	 */
	public Set<T> getAffectedTargets() {
		affectedTargets = new HashSet<T>();
		gatherTargets(addedMappings, affectedTargets);
		gatherTargets(deletedMappings, affectedTargets);
		for (Pair<M, M> mPair : modifiedMappings) {
			affectedTargets.add((T) mPair.getFirst().getTarget());
			affectedTargets.add((T) mPair.getSecond().getTarget());
		}
		return affectedTargets;
	}

	/**
	 * Gets all the targets of the old mappings.
	 *
	 * @return a set of the targets of the old mappings.
	 */
	public Set<T> getOldTargets() {
		Set<T> targets = new HashSet<T>();
		gatherTargets(oldMappings, targets);
		return targets;
	}

	/**
	 * Gets all the targets of the new mappings.
	 *
	 * @return a set of the targets of the new mappings.
	 */
	public Set<T> getNewTargets() {
		Set<T> targets = new HashSet<T>();
		gatherTargets(newMappings, targets);
		return targets;
	}

	private void gatherTargets(Collection<M> mappings, Set<T> targetsCollector) {
		for (M m : mappings) {
			targetsCollector.add((T) m.getTarget());
		}
	}

	/**
	 * Can be overridden by a subclass to set the defaults on a automatically generated default mapping. This default implementation does nothing.
	 *
	 * @param deployment        the deployment for which this default mapping was created
	 * @param newDefaultMapping the newly created default mapping
	 */
	public void setDefaults(Deployment deployment, M newDefaultMapping) {
		// no-op
	}

	@SuppressWarnings("rawtypes")
	private List<M> calculateMappings(Deployment deployment) {
		List<M> mappings = new ArrayList<M>();
		if (deployment != null) {
			Collection<S> sources = deployment.getSource().getMembersOfType(sourceCiClass);
			for (S eachSource : sources) {
				/*
				 * Looks for *any* explicit mappings of the source, not just mappings of this type. Otherwise, you'd get default mappings for an Ear even if you
				 * had already mapped it with an EarMapping (i.e. not a *WasEarMapping*).
				 */
				List<? extends Mapping> explicitMappings = deployment.getMappingsForSource(Mapping.class, eachSource);
				if (!explicitMappings.isEmpty()) {
					filterExplicitMappings(mappings, explicitMappings);
				} else if (applyDefaultMappings) {
					generateDefaultMappingsForSource(mappings, eachSource, deployment);
				}
			}
		}
		return mappings;
	}

	private void generateDefaultMappingsForSource(List<M> mappings, S source, Deployment deployment) {
		for (T eachTarget : deployment.getTarget().getMembersOfType(targetCiClass)) {
			M newMapping;
			try {
				newMapping = mappingCiClass.newInstance();
			} catch (InstantiationException e) {
				throw new RuntimeException(e);
			} catch (IllegalAccessException e) {
				throw new RuntimeException(e);
			}
			newMapping.setLabel("Default mapping from " + source + " to " + eachTarget);
			newMapping.setTarget(eachTarget);
			newMapping.setSource(source);
			setDefaults(deployment, newMapping);
			mappings.add(newMapping);
		}
	}

	// ignore mappings of a type not handled by the current mapper

	@SuppressWarnings("rawtypes")
	private void filterExplicitMappings(List<M> mappings, List<? extends Mapping> explicitMappings) {
		for (Mapping each : explicitMappings) {
			if (mappingCiClass.isInstance(each) && targetCiClass.isInstance(each.getTarget())) {
				applySourcePropertyOverrides(each);
				mappings.add((M) each);
			}
		}
	}

	// applies overrides "in place", i.e. *modifies* the sources of the mappings!

	@SuppressWarnings("rawtypes")
	private void applySourcePropertyOverrides(Mapping mapping) {
		if (mapping instanceof SourcePropertyOverridingMapping) {
			Serializable copyOfsourceWithPropertiesOverridden = applyPropertyOverrides(mapping.getSource(),
					((SourcePropertyOverridingMapping) mapping).getSourcePropertyOverrides());
			mapping.setSource(copyOfsourceWithPropertiesOverridden);
		}
	}

	private static Serializable applyPropertyOverrides(Serializable source, List<KeyValuePair> sourcePropertyOverrides) {
		Map<String, String> overrides = new HashMap<String, String>();
		if (sourcePropertyOverrides != null) {
			for (KeyValuePair override : sourcePropertyOverrides) {
				overrides.put(override.getKey(), override.getValue());
			}
		}
		try {
			return ConfigurationItemReflectionUtils.overrideProperties(source, overrides);
		} catch (IllegalArgumentException exception) {
			throw new ResolutionException(String.format("Invalid property override specified for %s in %s: %s", source, overrides, exception.getMessage()),exception);
		}
	}

	private List<M> substractSimilarMappings(final List<M> lhs, final List<M> rhs) {
		List<M> result = new ArrayList<M>();
		for (final M l : lhs) {
			if (!CollectionUtils.exists(rhs, new Predicate() {
				public boolean evaluate(final Object r) {
					return ((Mapping<S, T>) l).isSimilar((Mapping<S, T>) r);
				}
			})) {
				result.add(l);
			}
		}

		return result;
	}

	private List<Pair<M, M>> findModifiedMappings(final List<M> oldMappings, final List<M> newMappings) {
		List<Pair<M, M>> modifiedMappings = new ArrayList<Pair<M, M>>();
		for (M l : oldMappings) {
			for (M r : newMappings) {
				Mapping<S, T> correctlyCapturedL = (Mapping<S, T>) l;
				Mapping<S, T> correctlyCapturedR = (Mapping<S, T>) r;
				boolean mappingsAreSimilar = correctlyCapturedL.isSimilar(correctlyCapturedR);
				boolean mappingsAreIdentical = correctlyCapturedL.isIdentical(correctlyCapturedR);
				if (mappingsAreSimilar && !mappingsAreIdentical) {
					modifiedMappings.add(new Pair<M, M>(l, r));
					break;
				}
			}
		}
		return modifiedMappings;
	}


	protected Function<Pair<M, M>, M> FIRST_MAPPING = new Function<Pair<M, M>, M>() {
		public M apply(Pair<M, M> from) {
			return from.getFirst();
		}
	};

	protected Function<Pair<M, M>, M> SECOND_MAPPING = new Function<Pair<M, M>, M>() {
		public M apply(Pair<M, M> from) {
			return from.getSecond();
		}
	};

}
