package com.xebialabs.deployit.steps;

import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.StepExecutionContext;
import com.xebialabs.deployit.StepExecutionContextCallbackHandler;
import com.xebialabs.deployit.StepExecutionContextListener;
import com.xebialabs.deployit.ci.Database;
import com.xebialabs.deployit.ci.Host;
import com.xebialabs.deployit.ci.OperatingSystemFamily;
import com.xebialabs.deployit.ci.artifact.SqlFolder;
import com.xebialabs.deployit.ci.artifact.mapping.PlaceholderFormat;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.hostsession.*;

import java.util.Map;

@SuppressWarnings("serial")
public abstract class AbstractDatabaseStep implements Step {

	protected final Database database;
	protected final SqlFolder source;
	private final Map<String, String> properties;
	private final PlaceholderFormat placeholderFormat;


	public AbstractDatabaseStep(SqlFolder source, Database database, Map<String, String> properties, PlaceholderFormat placeholderFormat) {
		this.source = source;
		this.database = database;
		this.properties = properties;
		this.placeholderFormat = placeholderFormat;
	}

	protected HostSession getHostSession(StepExecutionContext ctx) {
		Host host = database.getHost();
		String key = "HSC_" + host.getLabel();
		@SuppressWarnings("unchecked")
        AttributeValue<HostSession> hsc = (AttributeValue<HostSession>) ctx.getAttribute(key);
		if (hsc == null) {
			hsc = new AttributeValue<HostSession>(host.getHostSession(), new ContextDestroyedHandler<HostSession>() {
				public void close(HostSession hostSession) {
					hostSession.close();
				}
			});
			ctx.setAttribute(key, hsc);
		}
		return hsc.get();
	}

	protected String getSqlRunnerPath(StepExecutionContext ctx) {
		String sqlRunnerPath = (String) ctx.getAttribute(database.getLabel());
		if (sqlRunnerPath == null) {
			final GenerateSqlRunnerScriptStep runnerScriptStep = new GenerateSqlRunnerScriptStep();
			ctx.logOutput(runnerScriptStep.getDescription());
			runnerScriptStep.execute(ctx);
			sqlRunnerPath = (String) ctx.getAttribute(database.getLabel());
		}
		ctx.logOutput("SqlRunner file is " + sqlRunnerPath);
		return sqlRunnerPath;
	}

	protected HostFile getRemoteLocation(StepExecutionContext ctx) {
		final String key = "RL" + source.getLabel();
		@SuppressWarnings("unchecked")
        AttributeValue<HostFile> avRemote = (AttributeValue<HostFile>) ctx.getAttribute(key);
		if (avRemote == null) {

			final HostSession sourceSession = Host.getLocalHost().getHostSession();
			try {
				final HostFile sourceHostDir = sourceSession.getFile(source.getLocation());
				final HostFile destinationDir = getHostSession(ctx).getTempFile(source.getLabel().replace(' ', '_'), ".sql");
				destinationDir.mkdirs();

				String description = "Copy " + source.getLocation() + " from " + Host.getLocalHost() + " to " + destinationDir + " on " + database.getHost();
				if (this.properties != null && !properties.isEmpty()) {
					description += " while replacing property values using " + placeholderFormat + " placeholder format";
				}
				ctx.logOutput(description);

				HostFileUtils.copy(sourceHostDir, destinationDir, getTransformer());
				avRemote = new AttributeValue<HostFile>(destinationDir);
				ctx.setAttribute(key, avRemote);
			} finally {
				sourceSession.close();
			}


		}
		return avRemote.get();
	}

	private HostFileInputStreamTransformer getTransformer() {
		if (properties == null)
			return null;
		if (properties.isEmpty())
			return null;
		return new LenientTemplateResolvingHostFileInputTransformer(properties, placeholderFormat);
	}


	interface ContextDestroyedHandler<T> {
		public void close(T t);
	}

	private class AttributeValue<T> implements StepExecutionContextListener {
		private final T value;
		private final ContextDestroyedHandler<T> handler;

		public AttributeValue(T value) {
			this(value, null);
		}

		public AttributeValue(T value, ContextDestroyedHandler<T> handler) {
			this.value = value;
			this.handler = handler;
		}

		public T get() {
			return value;
		}

		public void contextDestroyed() {
			if (handler != null && value != null)
				handler.close(value);
		}
	}

	class GenerateSqlRunnerScriptStep implements Step {

		protected OperatingSystemFamily osFamily;
		protected String suffix;
		protected String env;

		public GenerateSqlRunnerScriptStep() {
			this.osFamily = database.getHost().getOperatingSystemFamily();
			this.suffix = osFamily.getScriptExtension();
			this.env = (osFamily == OperatingSystemFamily.UNIX ? "export" : "set");
		}

		public String getDescription() {
			return "Generate SqlRunner file remote command on " + database.getHost() + ".";
		}

		public boolean execute(StepExecutionContext ctx) {
			HostSession s = getHostSession(ctx);
			try {
				CommandExecutionCallbackHandler handler = new StepExecutionContextCallbackHandler(ctx);

				ctx.logOutput("Generate the sql runner file on " + database.getHost());
				HostFile sqlRunnerFile = s.getTempFile("sqlRunner-" + database.getLabel().replace(' ', '_'), suffix);

				sqlRunnerFile.getParentFile().mkdirs();

				String currentPassword = database.getPassword();
				database.setPassword("*********");
				ctx.logOutput(database.getCommand());

				database.setPassword(currentPassword);
				final String command = database.getCommand();
				HostFileUtils.putStringToHostFile(command, sqlRunnerFile);

				if (osFamily.equals(OperatingSystemFamily.UNIX))
					s.execute(handler, "chmod", "+x", sqlRunnerFile.getPath());

				ctx.setAttribute(database.getLabel(), sqlRunnerFile.getPath());
				return true;

			} catch (RuntimeIOException exc) {
				throw new RuntimeIOException("Cannot generate bulk file remote command on " + database.getHost(), exc);
			}
		}
	}

}
