/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.util;

import com.xebialabs.deployit.ResolutionException;
import com.xebialabs.deployit.ci.artifact.War;
import com.xebialabs.deployit.ci.mapping.EjbReference;
import com.xebialabs.deployit.ci.mapping.ResourceReference;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.List;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

/**
 * Utility class to extract the list of resource references, resource environment references and EJB references from the web.xml file present in an War.
 */
public class WarFileInfoExtractor {

	public static List<ResourceReference> getResourceReferences(War war) {
		List<ResourceReference> resourceReferences = new ArrayList<ResourceReference>();
		NodeList resourceRefNodes = getNodesWithTag(war, "resource-ref");
		if (resourceRefNodes != null) {
			for (int i = 0; resourceRefNodes != null && i < resourceRefNodes.getLength(); i++) {
				Element resourceRefNode = (Element) resourceRefNodes.item(i);
				String name = findName(resourceRefNode, "res-ref-name");
				String type = findName(resourceRefNode, "res-type");
				if (StringUtils.isNotBlank(name)) {
					resourceReferences.add(new ResourceReference(name, type, ""));
				}
			}
		}
		return resourceReferences;
	}

	public static List<ResourceReference> getResourceEnvironmentEntryReferences(War war) {
		List<ResourceReference> resourceEnvironmentReferences = new ArrayList<ResourceReference>();
		NodeList resourceEnvRefNodes = getNodesWithTag(war, "resource-env-ref");
		if (resourceEnvRefNodes != null) {
			for (int i = 0; resourceEnvRefNodes != null && i < resourceEnvRefNodes.getLength(); i++) {
				Element resourceEnvRefNode = (Element) resourceEnvRefNodes.item(i);
				String name = findName(resourceEnvRefNode, "resource-env-ref-name");
				String type = findName(resourceEnvRefNode, "resource-env-ref-type");
				if (name != null && type != null) {
					resourceEnvironmentReferences.add(new ResourceReference(name, type, ""));
				}
			}
		}
		return resourceEnvironmentReferences;
	}

	public static List<EjbReference> getEjbReferences(War war) {
		List<EjbReference> ejbReferencesAndLocals = new ArrayList<EjbReference>();
		NodeList ejbLocalRefNodes = getNodesWithTag(war, "ejb-local-ref");
		if (ejbLocalRefNodes != null) {
			for (int i = 0; i < ejbLocalRefNodes.getLength(); i++) {
				Element ejbLocalRefNode = (Element) ejbLocalRefNodes.item(i);
				String name = findName(ejbLocalRefNode, "ejb-ref-name");
				String local = findName(ejbLocalRefNode, "local");
				if (name != null && local != null) {
					ejbReferencesAndLocals.add(new EjbReference(name, local, ""));
				}
			}
		}
		return ejbReferencesAndLocals;
	}

	private static NodeList getNodesWithTag(War war, String tag) {
		InputStream is = null;
		try {
			JarFile jarFile = new JarFile(new File(war.getLocation()));
			JarEntry jarEntry = jarFile.getJarEntry("WEB-INF/web.xml");
			if (jarEntry != null) {
				is = jarFile.getInputStream(jarEntry);
				DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
				DocumentBuilder db = dbf.newDocumentBuilder();
				disableDtdValidation(db);
				Document doc = db.parse(new InputSource(is));
				NodeList resourceRefNodes = doc.getElementsByTagName(tag);
				is.close();
				jarFile.close();
				return resourceRefNodes;
			}
			return null;
		} catch (Exception e) {
			throw new ResolutionException("Problems occurred while trying to parse " + war.getName() + " for finding value of tag:" + tag +" (location "+war.getLocation()+")", e);
		} finally {
			IOUtils.closeQuietly(is);
		}
	}

	private static String findName(Element element, String tagnameOfChild) {
		NodeList refNameElements = element.getElementsByTagName(tagnameOfChild);
		if (refNameElements.getLength() == 1) {
			Node refNameElement = refNameElements.item(0);
			return refNameElement.getFirstChild().getNodeValue();
		}
		return null;
	}

	private static void disableDtdValidation(DocumentBuilder db) {
		// disable the dtd validation by setting a custom entity resolver
		db.setEntityResolver(new EntityResolver() {
			public InputSource resolveEntity(String publicId, String systemId) throws SAXException, IOException {
				System.out.println("Ignoring " + publicId + ", " + systemId);
				return new InputSource(new StringReader(""));
			}
		});
	}
}
