/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.ci.artifact.mapping;

import java.io.Serializable;
import java.util.List;

import com.google.common.collect.Lists;
import com.xebialabs.deployit.ConfigurationItemProperty;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.ci.artifact.War;
import com.xebialabs.deployit.ci.mapping.EjbReference;
import com.xebialabs.deployit.ci.mapping.ResourceReference;
import com.xebialabs.deployit.util.WarFileInfoExtractor;

/**
 * A mapping of an {@link War} to any kind of middleware.
 */
@SuppressWarnings("serial")
public abstract class WarMapping<T extends Serializable> extends DeployableArtifactMapping<War, T> {

	@ConfigurationItemProperty(description="Virtual host to deploy to", required = false)
	private String virtualHost;

	@ConfigurationItemProperty(description="Context root to deploy to", required = false)
	private String contextRoot;
	
	@ConfigurationItemProperty(description = "Specifies the mapping from resource references jndi names and types used in the web.xml to resource references jndi names available in middleware", required = false, category = "Jndi references")
	private List<ResourceReference> resourceReferences = Lists.newArrayList();
	
	@ConfigurationItemProperty(description = "Specifies the mapping from resource environment references jndi names and types used in the web.xml to resource references jndi names available in middleware", required = false, category = "Jndi references")
	private List<ResourceReference> resourceEnvironmentEntryReferences = Lists.newArrayList();
	
	@ConfigurationItemProperty(description = "Specifies the mapping from ejb reference jndi names and locals used in the web.xml to bean jndi names available in middleware", required = false, category = "Jndi references")
	private List<EjbReference> ejbReferences = Lists.newArrayList();

	public WarMapping() {
		// default constructor
	}

	public WarMapping(War source, T target, String virtualHost, String contextRoot) {
		super(source, target);
		this.virtualHost = virtualHost;
		this.contextRoot = contextRoot;
	}
	
	@Override
	public void postInit(Deployment deployment) {
		this.setVirtualHost(deployment.getVhostDefinition());
		this.setContextRoot(getSource().getName());
		this.resourceReferences = WarFileInfoExtractor.getResourceReferences(getSource());
		this.resourceEnvironmentEntryReferences = WarFileInfoExtractor.getResourceEnvironmentEntryReferences(getSource());
		this.ejbReferences = WarFileInfoExtractor.getEjbReferences(getSource());
	}

	public String getVirtualHost() {
		return virtualHost;
	}

	public void setVirtualHost(String virtualHost) {
		this.virtualHost = virtualHost;
	}

	public String getContextRoot() {
		return contextRoot;
	}

	public void setContextRoot(String contextRoot) {
		this.contextRoot = contextRoot;
	}
	
	public List<ResourceReference> getResourceEnvironmentEntryReferences() {
		return resourceEnvironmentEntryReferences;
	}

	public List<ResourceReference> getResourceReferences() {
		return resourceReferences;
	}

	public List<EjbReference> getEjbReferences() {
		return ejbReferences;
	}

}