/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.mapper;

import java.io.Serializable;
import java.util.List;

import com.xebialabs.deployit.Change;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.ci.mapping.Mapping;

/**
 * Subclass of {@link StepGeneratingMapper} that can also generate modification steps for modified mappings.
 * 
 * @param <S>
 *            the type of the source artifact/resource.
 * @param <M>
 *            the type of the mapping.
 * @param <T>
 *            the type of the target middleware.
 */
public abstract class ModificationSupportingStepGeneratingMapper<S extends Serializable, M extends Mapping<? super S, ? super T>, T extends Serializable> extends
		StepGeneratingMapper<S, M, T> {

	/**
	 * Creates a <tt>ModificationSupportingStepGeneratingMapper</tt> for the specified deployment change.
	 * 
	 * @param change
	 *            the deployment change for which to calculate mappings.
	 * @param applyDefaultMappings
	 *            default mappings will be applied iff <code>true</code>
	 */
	protected ModificationSupportingStepGeneratingMapper(Change<Deployment> change, boolean applyDefaultMappings) {
		super(change, applyDefaultMappings);
	}

	/**
	 * Creates a <tt>ModificationSupportingStepGeneratingMapper</tt> for the specified deployment change that will apply
	 * default mappings.
	 * 
	 * @param change
	 *            the deployment change for which to calculate mappings.
	 */
	protected ModificationSupportingStepGeneratingMapper(Change<Deployment> change) {
		this(change, true);
	}

	/**
	 * Overrides
	 * {@link StepGeneratingMapper#generateAdditionStepsForModifiedMapping(Serializable, Mapping, Serializable, List)}
	 * to not generate any steps.
	 */
	@Override
	protected final void generateAdditionStepsForModifiedMapping(S newSource, M newMapping, T newTarget, List<Step> steps) {
		// no-op
	}

	/**
	 * Overrides
	 * {@link StepGeneratingMapper#generateDeletionStepsForModifiedMapping(Serializable, Mapping, Serializable, List)}
	 * to not generate any steps.
	 */
	@Override
	protected final void generateDeletionStepsForModifiedMapping(S oldSource, M oldMapping, T oldTarget, List<Step> steps) {
		// no-op
	}

	/**
	 * Generates all the steps that will modify artifacts/resources in the middleware.
	 * 
	 * @param steps
	 *            the list of steps to which the generated steps are to be added
	 */
	@SuppressWarnings("unchecked")
	public final void generateModificationSteps(List<Step> steps) {
		for (Pair<M, M> m : modifiedMappings) {
			M oldM = m.getFirst();
			M newM = m.getSecond();
			generateModificationStepsForModifiedMapping((S) oldM.getSource(), oldM, (T) oldM.getTarget(), (S) newM.getSource(), newM, (T) newM.getTarget(),
					steps);
		}
	}

	/**
	 * Must be implemented by a subclass to generate steps that delete an artifact/a resource from the middleware when a
	 * mapping is deleted.
	 * 
	 * @param oldMappingSource
	 *            the old source of the modified mapping. The value is identical to <tt>oldMapping.getSource()</tt> but
	 *            it is provided separately so that the implementor of this method does not have to cast it.
	 * @param oldVersionOfModifiedMapping
	 *            the old version of the modified mapping
	 * @param oldMappingTarget
	 *            the target of the old mapping. The value is identical to <tt>oldMapping.getTarget()</tt> but it is
	 *            provided separately so that the implementor of this method does not have to cast it.
	 * @param newMappingSource
	 *            the new source of the mapping. The value is identical to
	 *            <tt>newVersionOfModifiedMapping.getSource()</tt> but it is provided separately so that the implementor
	 *            of this method does not have to cast it.
	 * @param newVersionOfModifiedMapping
	 *            the new mapping
	 * @param newMappingTarget
	 *            the new target of the mapping. The value is identical to
	 *            <tt>newVersionOfModifiedMapping.getTarget()</tt> but it is provided separately so that the implementor
	 *            of this method does not have to cast it.
	 * @param steps
	 *            the list of steps to which the generated steps are to be added
	 */
	protected void generateModificationStepsForModifiedMapping(S oldMappingSource, M oldVersionOfModifiedMapping, T oldMappingTarget, S newMappingSource,
			M newVersionOfModifiedMapping, T newMappingTarget, List<Step> steps) {
		// no-op
	}

}
