package com.xebialabs.deployit.hostsession.ssh;

import com.jcraft.jsch.ChannelExec;
import com.jcraft.jsch.JSchException;
import com.jcraft.jsch.Session;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.hostsession.CmdLine;
import com.xebialabs.deployit.hostsession.HostFile;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

public class SshScpProtocolHelper {

    private String command;

    private Session session;

    private ChannelExec channel;

    private InputStream channelIn;

    private OutputStream channelOut;

    private static final String CHANNEL_PURPOSE = " (for SCP stream)";

    private SshHostSession sshHostSession;
    private String remotePath;

    public SshScpProtocolHelper(SshHostSession sshHostSession, String remotePath) {
        this.sshHostSession = sshHostSession;
        this.remotePath = remotePath;
    }

    public void open(String scpOptions) {
        try {
            // connect to SSH and start scp in sink mode
            session = sshHostSession.openSession(CHANNEL_PURPOSE);
            channel = (ChannelExec) session.openChannel("exec");
            // no password in this command, so use 'false'
            command = sshHostSession.encodeCommandLineForExecution(CmdLine.asCommandLine("scp", "-" + scpOptions, remotePath));
            channel.setCommand(command);
            channelIn = channel.getInputStream();
            channelOut = channel.getOutputStream();
            channel.connect();
            logger.info("Executing remote command \"" + command + "\" on " + this + " to open SCP stream");
            logger.info("Opened SCP stream to remote file " + remotePath);
        } catch (IOException exc) {
            throw new RuntimeIOException("Cannot open SCP stream to remote file " + remotePath, exc);
        } catch (JSchException exc) {
            throw new RuntimeIOException("Cannot open SCP stream to remote file " + remotePath, exc);
        }
    }

    public void readAck() {
        readAck(0);
    }

    public void readAck(int expectedChar) {
        if (logger.isDebugEnabled())
            logger.debug("Reading ACK");

        int c = SshStreamUtils.checkAck(channelIn);
        if (c != expectedChar) {
            throw new RuntimeIOException("Protocol error on SCP stream to read remote file " + remotePath + " (remote scp command returned acked with a \"" + c + "\")");
        }
    }

    public void sendFilePreamble(HostFile file, long length) throws IOException {
        String preamble = "C0644 " + length + " " + getFileName(file);

        if (logger.isDebugEnabled())
            logger.debug("Sending file preamble \"" + preamble + "\"");
        preamble += "\n";

        channelOut.write(preamble.getBytes());
        channelOut.flush();
    }

    private String getFileName(HostFile file) {
        if (file.getPath().lastIndexOf('/') > 0) {
            return file.getPath().substring(file.getPath().lastIndexOf('/') + 1);
        } else {
            return file.getPath();
        }
    }

    public void sendDirectoryPostscript(HostFile dir) throws IOException {
        String postscript = "E\n";
        channelOut.write(postscript.getBytes());
        channelOut.flush();

    }

    public void sendDirectoryPreamble(HostFile dir) throws IOException {
        String preamble = "D0755 0 " + getFileName(dir);
        if (logger.isDebugEnabled())
            logger.debug("Sending dir preamble \"" + preamble + "\"");
        preamble += "\n";

        channelOut.write(preamble.getBytes());
        channelOut.flush();
    }

    public void sendAck() throws IOException {
        if (logger.isDebugEnabled())
            logger.debug("Sending ACK");

        channelOut.write('\0');
        channelOut.flush();
    }

    public void write(InputStream is) throws IOException {
        IOUtils.copy(is, channelOut);
    }

    public void write(byte[] b) throws IOException {
        channelOut.write(b);
    }

    public void write(byte[] b, int off, int len) throws IOException {
        channelOut.write(b, off, len);
    }

    public void write(int b) throws IOException {
        channelOut.write(b);
    }

    public int close() {
        // get return code from remote scp
        int res = 0;
        if (channel != null) {
             // close output channel to force remote scp to quit
            IOUtils.closeQuietly(channelOut);
            res = SshHostSession.waitForExitStatus(channel, command);
            IOUtils.closeQuietly(channelIn);
            channel.disconnect();
        }

        sshHostSession.disconnectSession(session, CHANNEL_PURPOSE);

        return res;
    }

    public InputStream getChannelIn() {
        return channelIn;
    }

    private Logger logger = LoggerFactory.getLogger(SshScpProtocolHelper.class);
}
