package com.xebialabs.deployit.hostsession.common;

import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.hostsession.HostFile;
import com.xebialabs.deployit.hostsession.HostFileInputStreamTransformer;
import com.xebialabs.deployit.hostsession.HostFileInputStreamTransformer.TransformedInputStream;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.Stack;

/**
 * HostFile copy utility that uses only the input and output streams
 * exposed by the HostFile to perform the copying action.
 */
public final class HostFileCopier extends HostFileDirectoryWalker {

    private static final String SOURCE = "Source";
    private static final String DESTINATION = "Destination";

    private Stack<HostFile> dstDirStack = new Stack<HostFile>();
    private HostFile srcDir;
    private HostFileInputStreamTransformer transformer;

    private HostFileCopier(HostFile srcDir, HostFile dstDir, HostFileInputStreamTransformer transformer) {
        dstDirStack.push(dstDir);
        this.srcDir = srcDir;
        HostFileCopier.checkDirectoryExists(srcDir, SOURCE);
        this.transformer = transformer;
    }

    @Override
    protected void handleDirectoryStart(HostFile scrDir, int depth) throws IOException {
        HostFile dstDir = getCurrentDestinationDir();
        if (depth != ROOT) {
            dstDir = createSubdirectoryAndMakeCurrent(dstDir,scrDir.getName());
        }

        if (dstDir.exists()) {
            HostFileCopier.checkReallyIsADirectory(dstDir, DESTINATION);
            if (logger.isDebugEnabled())
                logger.debug("About to copy files into existing directory " + dstDir);
        } else {
            if (logger.isDebugEnabled())
                logger.debug("Creating destination directory " + dstDir);
            dstDir.mkdir();
        }
    }

    private HostFile createSubdirectoryAndMakeCurrent(HostFile parentDir, String subdirName) {
         HostFile subdir = parentDir.getFile(subdirName);
         dstDirStack.push(subdir);
        return subdir;
    }

    private void startCopy() {
        walk(srcDir);
    }

    private HostFile getCurrentDestinationDir() {
        return dstDirStack.peek();
    }

    @Override
    protected void handleFile(HostFile srcFile, int depth) throws IOException {
        HostFile dstFile = getCurrentDestinationDir().getFile(srcFile.getName());
        HostFileCopier.copyFile(srcFile, dstFile, transformer);
    }

    @Override
    protected void handleDirectoryEnd(HostFile directory, int depth) throws IOException {
        if (depth != ROOT) {
            dstDirStack.pop();
        }
    }

    /**
     * Copies a file or directory.
     *
     * @param src         the source file or directory.
     * @param dst         the destination file or directory. If it exists it must be of the same type as the source. Its parent directory must exist.
     * @param transformer Transforms the inputstream of the sourcefile, can supply null
     * @throws RuntimeIOException if an I/O error occurred
     */
    public static void copy(HostFile src, HostFile dst, HostFileInputStreamTransformer transformer) {
        if (src.isDirectory()) {
            copyDirectory(src, dst, transformer);
        } else {
            copyFile(src, dst, transformer);
        }
    }

    /**
     * Copies a directory recursively.
     *
     * @param srcDir      the source directory. Must exist and must not be a directory.
     * @param dstDir      the destination directory. May exists but must a directory. Its parent directory must exist.
     * @param transformer Transforms the inputstream of the sourcefile, can be null
     * @throws RuntimeIOException if an I/O error occurred
     */
    public static void copyDirectory(HostFile srcDir, HostFile dstDir, HostFileInputStreamTransformer transformer) throws RuntimeIOException {
        HostFileCopier dirCopier = new HostFileCopier(srcDir, dstDir, transformer);
        dirCopier.startCopy();
    }


    /**
     * Copies a regular file.
     *
     * @param srcFile     the source file. Must exists and must not be a directory.
     * @param dstFile     the destination file. May exists but must not be a directory. Its parent directory must exist.
     * @param transformer Transforms the inputstream of the sourcefile, can be null
     * @throws com.xebialabs.deployit.exception.RuntimeIOException
     *          if an I/O error occurred
     */
    public static void copyFile(HostFile srcFile, HostFile dstFile, HostFileInputStreamTransformer transformer) throws RuntimeIOException {
        checkFileExists(srcFile, SOURCE);
        checkReallyIsAFile(dstFile, DESTINATION);

        if (logger.isDebugEnabled()) {
            if (dstFile.exists())
                logger.debug("About to overwrite existing file " + dstFile);
            logger.debug("Copying contents of file " + srcFile + " to " + dstFile);
        }

        copyStreamToFile(transform(srcFile, transformer), dstFile);
    }

    private static void copyStreamToFile(TransformedInputStream in, HostFile dstFile) {
        try {
            dstFile.put(in, in.length());
        } finally {
            IOUtils.closeQuietly(in);
        }
    }

    /**
     * Applies the specified transformer to the source file. If the transformer is null,
     * the inputstream from the source file is used directly.
     *
     * @param file that must be transformed.  Must exists.
     * @param transformer transformer Transforms the inputstream of the sourcefile, can be null
     * @return TransformedInputStream as determined by the transformer.
     */
    public static TransformedInputStream transform(HostFile file, HostFileInputStreamTransformer transformer) {
        if (transformer == null) {
            return new TransformedInputStream(file.get(), file.length());
        } else {
            return transformer.transform(file);
        }
    }

    /**
     * Assert that the file must exist and it is not a directory.
     * @param file to check.
     * @param sourceDescription to prepend to error message.
     * @throws RuntimeIOException if file does not exist or is a directory.
     */
    public static void checkFileExists(HostFile file, String sourceDescription) {
        if (!file.exists()) {
            throw new RuntimeIOException(sourceDescription + " file " + file + " does not exist");
        }
        checkReallyIsAFile(file, sourceDescription);
    }

     /**
     * Assert that if a file exists, it is not a directory.
      *
     * @param file to check.
     * @param fileDescription to prepend to error message.
     * @throws RuntimeIOException if file is a directory.
     */
    public static void checkReallyIsAFile(HostFile file, String fileDescription) {
        if (file.exists() && file.isDirectory()) {
            throw new RuntimeIOException(fileDescription + " file " + file + " exists but is a directory");
        }
    }

     /**
     * Assert that the directory exists.
     *
     * @param dir is the directory to check.
     * @param dirDescription to prepend to error message.
     * @throws RuntimeIOException if directory does not exist or if it a flat file.
     */
    public static void checkDirectoryExists(HostFile dir, String dirDescription) {
        if (!dir.exists()) {
            throw new RuntimeIOException(dirDescription + " directory " + dir + " does not exist");
        }
        checkReallyIsADirectory(dir, dirDescription);
    }

    /**
     * Assert that if a file exists, it must be a directory.
     *
     * @param dir is the directory to check.
     * @param dirDescription to prepend to error message.
     * @throws RuntimeIOException if file is not a directory.
     */
    public static void checkReallyIsADirectory(HostFile dir, String dirDescription) {
        if (dir.exists() && !dir.isDirectory()) {
            throw new RuntimeIOException(dirDescription + " directory " + dir + " exists but is not a directory");
        }
    }


    private static Logger logger = LoggerFactory.getLogger(HostFileCopier.class);


}
