package com.xebialabs.deployit.hostsession.ssh;

import com.xebialabs.deployit.hostsession.common.HostFileCopier;
import com.xebialabs.deployit.hostsession.HostFile;
import com.xebialabs.deployit.hostsession.common.HostFileDirectoryWalker;
import com.xebialabs.deployit.hostsession.HostFileInputStreamTransformer;
import com.xebialabs.deployit.hostsession.HostFileInputStreamTransformer.TransformedInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;

public class SshScpDirectoryCopier extends HostFileDirectoryWalker {

    private HostFile scrDir;

    private SshScpHostFile targetDir;

    private HostFileInputStreamTransformer transformer;

    private SshScpProtocolHelper protocolHelper;


    protected SshScpDirectoryCopier(HostFile srcDir, SshScpHostFile dstDir, HostFileInputStreamTransformer transformer) {
        HostFileCopier.checkDirectoryExists(srcDir, "Source");
        this.scrDir = srcDir;
        this.targetDir = dstDir;
        this.transformer = transformer;
        this.protocolHelper = new SshScpProtocolHelper(dstDir.sshHostSession, dstDir.getPath());
    }

    @Override
    protected void handleDirectoryStart(HostFile directory, int depth) throws IOException {
        protocolHelper.sendDirectoryPreamble(directory);
        protocolHelper.readAck();
    }

    @Override
    protected void handleFile(HostFile file, int depth) throws IOException {
        TransformedInputStream transformedIS = HostFileCopier.transform(file, transformer);
        protocolHelper.sendFilePreamble(file, transformedIS.length());
        protocolHelper.readAck();
        protocolHelper.write(transformedIS);
        protocolHelper.sendAck();
    }


    @Override
    protected void handleDirectoryEnd(HostFile directory, int depth) throws IOException {
        protocolHelper.sendDirectoryPostscript(directory);
        protocolHelper.readAck();
    }

    public void startCopy() {
        try {
            openConnection();
            walk(scrDir);
        } finally {
            closeConnection();
        }
    }

    private void openConnection() {
        protocolHelper.open("rt");
        // perform SCP write protocol
        protocolHelper.readAck();
        logger.info("Opened SCP stream to write to remote dir " + targetDir);
    }

    private void closeConnection() {
        protocolHelper.close();
    }

    private Logger logger = LoggerFactory.getLogger(SshScpDirectoryCopier.class);
}
