package com.xebialabs.deployit.plugin.freemarker

import java.io.{IOException, StringReader, StringWriter}

import com.xebialabs.deployit.plugin.freemarker.ConfigurationHolder._
import com.xebialabs.overthere.RuntimeIOException
import freemarker.template.{Configuration, Template, TemplateException}
import grizzled.slf4j.Logging

private[freemarker] object Resolver extends Logging {

  val EXPRESSION_TOKEN = "${"
  val EXPRESSION_IF_TOKEN = "<#if"
  val NON_ESCAPED_COMMA = "(?<!\\\\),"


  def resolveExpression(expression: String, context: AnyRef): String = {
    if (!isResolvable(expression)) {
      return expression
    }
    val cfg: Configuration = configuration
    try {
      trace(s"Resolving expression $expression.")
      val template: Template = new Template("expression", new StringReader(expression), cfg)
      val sw: StringWriter = new StringWriter
      template.process(convertToJavaMap(context), sw)
      sw.toString
    }
    catch {
      case e: IOException => throw new RuntimeIOException(e)
      case e: TemplateException => throw new RuntimeIOException(e)
    }
  }

  def resolveExpressions(expressions: Iterable[String], context: Map[String, AnyRef]): Iterable[String] = {
    transformCollectionByResolvingExpression(expressions, context)
  }

  def resolveMapExpressions(expressions: Map[String, String], context: Map[String, AnyRef]): Map[String, String] = {
    val contextWithExpressions: Map[String, AnyRef] = context ++ expressions
    expressions.mapValues(expr => resolveWhilePossible(expr, contextWithExpressions))
  }

  private def resolveWhilePossible(expression: String, context: AnyRef): String = {
    if (isResolvable(expression))
      resolveWhilePossible(resolveExpression(expression, context), context)
    else
      expression
  }

  private def isResolvable(expression: String): Boolean = {
    expression != null && (expression.contains(EXPRESSION_TOKEN) || expression.contains(EXPRESSION_IF_TOKEN))
  }

  private[freemarker] def transformCollectionByResolvingExpression(expressions: Iterable[String], context: AnyRef) = {
    expressions.map(exp => resolveExpression(exp, context))
      .flatMap(resolvedExpr => resolvedExpr.split(NON_ESCAPED_COMMA))
      .map(_.trim).filterNot(_.isEmpty)
      .map(expr => expr.replaceAll("\\\\,", ",")).toIterable
  }
}
